<?php

namespace App\Http\Controllers\Admin;

use App\Exports\SolicitudExport;
use App\Http\Controllers\Controller;
use App\Http\Helpers\Consts;
use App\Mail\MessageReceived;
use App\Models\ComentarioMaterial;
use App\Models\Estatus;
use App\Models\Ingeniero;
use App\Models\Obra;
use App\Models\Producto;
use App\Models\RelSolicitudProductoComentarioMaterial;
use App\Models\solicitud_material\RelSolicitudCompraProductoPrecio;
use App\Models\SolicitudCompra;
use App\Models\SolicitudCompraProducto;
use App\Models\Unidad;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use DataTables;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use LynX39\LaraPdfMerger\Facades\PdfMerger;
use PDFSnappy;

class SolicitudComprasController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('can:solicitud_de_compras_index')->only('index');
        $this->middleware('can:solicitud_de_compras_create')->only('create', 'store');
        $this->middleware('can:solicitud_de_compras_edit')->only('edit', 'update');
        $this->middleware('can:solicitud_de_compras_destroy')->only('destroy');
        $this->middleware('can:solicitud_de_compras_show')->only('show');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();
        $user_role_root_dev = $user->hasrole('root-developer') ? 1 : 0;
        $user_role_super_usr_o_root_arq = $user->hasanyrole('super-usuario|root-arquitecto|root-contador') ? 1 : 0;
        //Role Gloria
        $role_ing_jefe_gloria = $user->hasRole('ingeniero-jefe'); // ? 1 : 0;
        //role ing
        $role_ing = $user->hasRole('ingeniero');

        $solicitud_de_compras_show = $user->can('solicitud_de_compras_show');
        $solicitud_de_compras_edit = $user->can('solicitud_de_compras_edit');
        $solicitud_de_compras_destroy = $user->can('solicitud_de_compras_destroy');
        $solicitud_de_compras_export  = $user->can('solicitud_de_compras_export');
        $solicitud_de_compras_send_email = $user->can('solicitud_de_compras_send_email');
        $solicitud_de_compras_authorized  = $user->can('solicitud_de_compras_authorized');
        $solicitud_de_compras_finalized  = $user->can('solicitud_de_compras_finalized');

        //Estatus: Por autorizar
        $estatus_por_autorizar = Estatus::where('clave', 'por_autorizar')->whereNull('deleted_at')->first();
        //Estatus: Autorizado
        $estatus_autorizado = Estatus::where('clave', 'autorizado')->whereNull('deleted_at')->first();
        //Estatus: finalizado
        $estatus_finalizado = Estatus::where('clave', 'finalizado')->whereNull('deleted_at')->first();

        return view('admin.solicitud_de_compras.lista',
            compact(
                'solicitud_de_compras_show',
                'solicitud_de_compras_edit',
                'solicitud_de_compras_destroy',
                'solicitud_de_compras_export',
                'solicitud_de_compras_send_email',
                'solicitud_de_compras_authorized',
                'solicitud_de_compras_finalized',
                'estatus_por_autorizar',
                'estatus_autorizado',
                'estatus_finalizado',
                'user_role_root_dev',
                'user_role_super_usr_o_root_arq',
                'role_ing_jefe_gloria',
                'role_ing',
            ));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $user = Auth::user();
        $user_can_solicitud_de_compras_create_select_ing_catalogo = $user->can('solicitud_de_compras_create_select_ing_catalogo');
        $ingeniero = null;

        if (!$user_can_solicitud_de_compras_create_select_ing_catalogo) {
            $ingeniero = Ingeniero::select('id', 'nombre')
                ->where('user_id', Auth::id())
                ->whereNull('deleted_at')
                ->first();
        }

        return view('admin.solicitud_de_compras.nuevo', compact( 'user', 'user_can_solicitud_de_compras_create_select_ing_catalogo', 'ingeniero'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $response = [];
        $dataAll = $request->all();
        $resultadoErrors = []; //Lista de Errores

        $validator = \Validator::make($dataAll, [
            'obra_id' => 'required|integer',
            'ingeniero_id' => 'required|integer',
            'productos_data' => 'required|array|min:1',
//            'productos_data.*.producto_id' => 'required|integer',
//            'productos_data.*.unidad_id' => 'required|integer',
//            'productos_data.*.cantidad' => 'required|integer',
        ]);

        if ($validator->fails()) {
            $response = [
                'status' => "ERROR",
                'error' => true,
                'message' => 'Campos obligatorios',
                'errors' => $validator->errors()->all()
            ];
        } else {

            $solicitud_compra = new SolicitudCompra();
            $solicitud_compra->obra_id = $dataAll['obra_id'];
            $solicitud_compra->ingeniero_id = $dataAll['ingeniero_id'];
            $solicitud_compra->observaciones = $dataAll['observaciones'];
            $solicitud_compra->fecha_solicitud = Carbon::now()->format('Y-m-d');

            if ($dataAll['fecha_entrega']){
                $solicitud_compra->fecha_entrega = (Carbon::createFromFormat('d/m/Y', $dataAll['fecha_entrega']))->format('Y-m-d');
            }

            $sinsolicitud = $dataAll['sinsolicitud'] ?? false;
            if (!$sinsolicitud){
                //Crear S.C. con Estatus Por autorizar
                $estatus_por_autorizar = Estatus::where('clave', 'por_autorizar')->whereNull('deleted_at')->first();
                $solicitud_compra->estatus_id = $estatus_por_autorizar->id;
            } else {
                //Crear S.C. con Estatus Autorizado desde una O.C.
                $estatus_autorizado = Estatus::where('clave', 'autorizado')->whereNull('deleted_at')->first();
                $solicitud_compra->estatus_id = $estatus_autorizado->id;
            }

            $solicitud_compra->emails = $dataAll['emails'];
            $solicitud_compra->email_asunto = $dataAll['email_asunto'];
            $solicitud_compra->email_contenido = $dataAll['email_contenido'];
            $solicitud_compra->clave_version = Consts::SOLICITUD_COMPRA_V2;

            if ($solicitud_compra->save()) {
                $foliador = $this->getFoliador($solicitud_compra->id);
                $solicitud_compra->codigo = $foliador;
                $solicitud_compra->codigo_nombre = 'solicitud-de-compra-'.$foliador;
                $solicitud_compra->update();

                //lista materiales
                $productos_data = $dataAll['productos_data'];
                //$sinsolicitud = $dataAll['sinsolicitud'] ?? false;
                if (!$sinsolicitud) {
                    //Crear S.C.
                    foreach ($productos_data as $producto) {
                        $solicitud_compra_producto = new SolicitudCompraProducto();
                        $solicitud_compra_producto->solicitud_compra_id = $solicitud_compra->id;
                        $solicitud_compra_producto->producto_id = intval($producto['1']);
                        $solicitud_compra_producto->unidad_id = $producto['3'];
                        $solicitud_compra_producto->cantidad = $producto['5'];
                        $solicitud_compra_producto->area_de_uso = $producto['6'];
                        $solicitud_compra_producto->marca = $producto['7'];
                        $solicitud_compra_producto->modelo = $producto['8'];
                        $solicitud_compra_producto->observaciones = $producto['9'];
                        $solicitud_compra_producto->num_duplicado_order = $producto['10'];
                        if (!$solicitud_compra_producto->save()) {
                            array_push($resultadoErrors, 'Ocurrio un error al intentar agregar material: ' . $producto['2']);
                        }
                    }
                } else {
                    //Crear S.C. desde una O.C. sin solicitud
                    foreach ($productos_data as $producto) {
                        $solicitud_compra_producto = new SolicitudCompraProducto();
                        $solicitud_compra_producto->solicitud_compra_id = $solicitud_compra->id;
                        $solicitud_compra_producto->producto_id = intval($producto['2']);
                        $solicitud_compra_producto->unidad_id = $producto['4'];
                        $solicitud_compra_producto->cantidad = $producto['6'];
                        $solicitud_compra_producto->area_de_uso = $producto['7'];
                        $solicitud_compra_producto->marca = $producto['8'];
                        $solicitud_compra_producto->modelo = $producto['9'];
                        $solicitud_compra_producto->observaciones = $producto['10'];
                        $solicitud_compra_producto->num_duplicado_order = $producto['16'];
                        if (!$solicitud_compra_producto->save()) {
                            array_push($resultadoErrors, 'Ocurrio un error al intentar agregar material: ' . $producto['2']);
                        }
                    }
                }

                if (!empty($resultadoErrors)) {
                    $response = [
                        'status' => "ERROR",
                        'error' => true,
                        'message' => 'Ocurrio un error al guardar la solicitud de compra.',
                        'errors' => $resultadoErrors
                    ];
                } else {
                    $response = [
                        'status' => "OK",
                        'error' => false,
                        'message' => 'Solicitud de compra guardada correctamente.',
                        'data' => $solicitud_compra
                    ];
                }
            } else {
                $response = [
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'Ocurrio un error al guardar la solicitud de compra.',
                    'errors' => 'New SolicitudCompra: error al guardar la solicitud de compra.'
                ];
            }
        }

        return response()->json($response);
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $clonSolicitud = filter_var(request('clonSolicitud', 0), FILTER_VALIDATE_BOOLEAN);

        $solicitud_compra = SolicitudCompra::with([
            'obra',
            'ingeniero',
            'estatus',
            'solicitudCompraProducto',
            'solicitudCompraProducto.producto',
            'solicitudCompraProducto.unidad',
            'solicitudCompraProducto.RelSolicitudProductoComentarioMaterial.comentariosMaterial',
        ])
            ->whereNull('deleted_at')
            ->where('id', $id)
            ->first();

        return view('admin.solicitud_de_compras.info', compact('solicitud_compra', 'clonSolicitud'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //Estatus: Autorizado
        $estatus_autorizado = Estatus::where('clave', 'autorizado')->whereNull('deleted_at')->first();
        //Estatus: finalizado
        $estatus_finalizado = Estatus::where('clave', 'finalizado')->whereNull('deleted_at')->first();

        $solicitud_compra = SolicitudCompra::with([
            'obra',
            'ingeniero',
            'estatus',
            'solicitudCompraProducto',
            'solicitudCompraProducto.producto',
            'solicitudCompraProducto.unidad',
            'solicitudCompraProducto.RelSolicitudProductoComentarioMaterial.comentariosMaterial',
        ])
            ->whereNull('deleted_at')
            ->where('id', $id)
            ->first();

        return view('admin.solicitud_de_compras.editar', compact('solicitud_compra', 'estatus_autorizado', 'estatus_finalizado'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $response = [];
        $dataAll = $request->all();
        $resultadoErrors = []; //Lista de Errores

        $validator = \Validator::make($dataAll, [
            //'obra_id' => 'required|integer',
            //'ingeniero_id' => 'required|integer',
            'productos_data' => 'required|array|min:1',
        ]);

        if ($validator->fails()) {
            $response = [
                'status' => "ERROR",
                'error' => true,
                'message' => 'Campos obligatorios',
                'errors' => $validator->errors()->all()
            ];
        } else {

            $solicitud_compra = SolicitudCompra::findOrFail($id);
            $solicitud_compra->observaciones = $dataAll['observaciones'];
            if ($dataAll['fecha_entrega']) {
                $solicitud_compra->fecha_entrega = (Carbon::createFromFormat('d/m/Y', $dataAll['fecha_entrega']))->format('Y-m-d');
            }

            $solicitud_compra->emails = $dataAll['emails'];
            $solicitud_compra->email_asunto = $dataAll['email_asunto'];
            $solicitud_compra->email_contenido = $dataAll['email_contenido'];

            if ($solicitud_compra->update()) {
                //Actualizar lista de materiales
                $productos_data = $dataAll['productos_data'];
                foreach ($productos_data as $producto) {
                    // $producto['1'] = Es la clave de registro
                    if (empty($producto['1'])) {
                        //Dar de alta Material Nuevo
                        $solicitud_compra_producto = new SolicitudCompraProducto();
                        $solicitud_compra_producto->solicitud_compra_id = $solicitud_compra->id;
                        $solicitud_compra_producto->producto_id = $producto['2'];
                        $solicitud_compra_producto->unidad_id = $producto['4'];
                        $solicitud_compra_producto->cantidad = $producto['6'];
                        $solicitud_compra_producto->area_de_uso = $producto['7'];
                        $solicitud_compra_producto->marca = $producto['8'];
                        $solicitud_compra_producto->modelo = $producto['9'];
                        $solicitud_compra_producto->observaciones = $producto['10'];
                        $solicitud_compra_producto->num_duplicado_order = $producto['13'];
                        if (!$solicitud_compra_producto->save()) {
                            array_push($resultadoErrors, 'Ocurrio un error al intentar agregar material: ' . $producto['3']);
                        }
                    } elseif (!empty($producto['1'])) {
                        //Actualizar Material existente
                        $material_actualizar = SolicitudCompraProducto::find($producto['1']);
                        if ($material_actualizar){
                            $material_actualizar->producto_id = $producto['2'];
                            $material_actualizar->unidad_id = $producto['4'];
                            $material_actualizar->cantidad = $producto['6'];
                            $material_actualizar->area_de_uso = $producto['7'];
                            $material_actualizar->marca = $producto['8'];
                            $material_actualizar->modelo = $producto['9'];
                            $material_actualizar->observaciones = $producto['10'];
                            $material_actualizar->num_duplicado_order = $producto['13'];
                            if (!$material_actualizar->update()) {
                                array_push($resultadoErrors, 'Ocurrio un error al intentar actualizar el material: ' . $producto['3']);
                            }
                        }
                    }

                }

                if (!empty($resultadoErrors)) {
                    $response = [
                        'status' => "ERROR",
                        'error' => true,
                        'message' => 'Ocurrió un error al guardar la solicitud de compra.',
                        'errors' => $resultadoErrors
                    ];
                } else {
                    $response = [
                        'status' => "OK",
                        'error' => false,
                        'message' => 'Solicitud de compra guardada correctamente.',
                        'data' => $solicitud_compra
                    ];
                }
            } else {
                $response = [
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'Ocurrió un error al actualizar la solicitud de compra.',
                    'errors' => 'Update SolicitudCompra: error al actualizar la solicitud de compra.'
                ];
            }
        }

        return response()->json($response);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $solicitud_compra = SolicitudCompra::with([
            'solicitudCompraProducto',
        ])->where('id', $id)->first();

        foreach ($solicitud_compra->solicitudCompraProducto as $material) {
            $material->delete();
        }

        $solicitud_compra->delete();

        return response()->json(['error' => false, 'message' => 'Registro eliminado.']);
    }

    public function get_Lista_solicitud_compras(Request $request)
    {
        if ($request->ajax()) {
            $solicitud_compras = SolicitudCompra::with([
                'obra',
                'ingeniero',
                'estatus',
                'solicitudCompraProducto'
            ])
                ->whereNull('deleted_at')
                ->orderBy('id', 'desc')->get();

            $solicitud_compras->map(function ($solicitudCompra) {
                //Lista materiales
                $conceptos_materiales = "";
                $lista_productos = $solicitudCompra->solicitudCompraProducto;
                foreach ($lista_productos as $key => $material) {
                    $conceptos_materiales .= $material->producto->nombre . ', ';
                }
                $conceptos_materiales = substr($conceptos_materiales, 0, -2);
                $conceptos_materiales .= ".";
                $solicitudCompra->conceptos_materiales = $conceptos_materiales;
            });

            return DataTables::of($solicitud_compras)->make(true);
        }
    }

    /**
     * Obetener lista de Solicitud de Compras solo y unicamnete con estatus = Autorizados
     * Para Crear las Nuevas Ordenes de Compra
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function get_Lista_solicitud_compras_Select(Request $request)
    {
        $solicitud_compras = [];
        //Estatus: Autorizado
        $autorizado = Estatus::where('clave', 'autorizado')->whereNull('deleted_at')->first();

        if ($request->has('search')) {
            $search = $request->search;

            $solicitud_compras = SolicitudCompra::with('obra', 'ingeniero', 'estatus')
                ->whereNull('deleted_at')
                ->where('codigo', 'LIKE', '%'.$search.'%')
                ->orWhere('codigo_nombre', 'LIKE', '%'.$search.'%')
                ->where('estatus_id', $autorizado->id)
                ->where('clave_version', Consts::SOLICITUD_COMPRA_V2)
                ->orderBy('id', 'desc')
                ->get();
        }else{
            $solicitud_compras = SolicitudCompra::with('obra', 'ingeniero', 'estatus')
                ->whereNull('deleted_at')
                ->where('estatus_id', $autorizado->id)
                ->where('clave_version', Consts::SOLICITUD_COMPRA_V2)
                ->orderBy('id', 'desc')->get();
        }

        if (!$solicitud_compras->isEmpty()) {
            $valid_tags = [];
            foreach ($solicitud_compras as $item) {
                //$valid_tags[] = ['id' => $item->id, 'text' => $item->codigo . ' - '. $item->meterial];
                if ($item->estatus_id === $autorizado->id && $item->clave_version === Consts::SOLICITUD_COMPRA_V2) {
                    $valid_tags[] = ['id' => $item->id, 'text' => $item->codigo];
                }
            }
            return response()->json($valid_tags);
        } else {
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'message' => 'No se encontraron resultados'
            ]);
        }
    }

    public function getProductoAndUnidadById($id_producto, $id_unidad)
    {
        try {
            $producto = Producto::where('id', $id_producto)
                ->select('id', 'nombre', 'codigo')
                ->first();
            $unidad = Unidad::where('id', $id_unidad)
                ->select('id', 'nombre')
                ->first();
            return response()->json([
                'error' => false,
                'mensajes' => 'Material encontrado',
                'producto' => $producto,
                'unidad' => $unidad
            ]);
        } catch (\Exception $ex) {
            return response()->json([
                'error' => true,
                'mensajes' => $ex->getMessage(),
            ]);
        }
    }

    private function getFoliador($ultimo_id = 0)
    {
        $foliador = 1;
        if ($ultimo_id > 0) {
            $foliador = str_pad($ultimo_id,  4, "0", STR_PAD_LEFT);
        } else {
            $foliador = str_pad($foliador,  4, "0", STR_PAD_LEFT);
        }
        return $foliador;
    }

    public function getSolicitudCompraById($id)
    {
        try {
            $solicitud_compra = SolicitudCompra::with([
                'obra',
                'ingeniero',
                'estatus',
                'solicitudCompraProducto',
                'solicitudCompraProducto.producto',
                'solicitudCompraProducto.unidad',
            ])
                ->whereNull('deleted_at')
                ->where('id', $id)
                ->first();

            return response()->json([
                'error' => false,
                'mensajes' => 'Solicitud de Compra encontrado',
                'solicitud_compra' => $solicitud_compra,
            ]);
        } catch (\Exception $ex) {
            return response()->json([
                'error' => true,
                'mensajes' => $ex->getMessage(),
            ]);
        }
    }

    public function print_pdf($id, $enviar_correo = 0)
    {
        try {
            $solicitud_compra = SolicitudCompra::with([
                'obra',
                'ingeniero',
                'estatus',
                'solicitudCompraProducto',
                'solicitudCompraProducto.producto',
                'solicitudCompraProducto.unidad',
                'solicitudCompraProducto.RelSolicitudProductoComentarioMaterial.comentariosMaterial',
            ])
                ->whereNull('deleted_at')
                ->where('id', $id)
                ->first();

//            $pdf = PDFSnappy::loadView('admin.solicitud_de_compras.print_pdf', compact('solicitud_compra'));
//            $pdf->setOption('header-spacing', 1);
//            $pdf->setOption('header-html', view('admin.prints.prints_header'));
//            $pdf->setOption('footer-html', view('admin.prints.prints_footer'));
//            $pdf->setOptions([
//                'margin-top'    => 15,
//                'margin-bottom' => 15,
//                'margin-right'  => 15,
//                'margin-left'   => 15,
//            ]);

            $orden_compra_solicitud_compra_clon_id = null;
            //generar lista pdf
            $pdf = App::make('dompdf.wrapper');
            $pdf->setOption(['dpi' => 110, 'defaultFont' => 'sans-serif', 'isPhpEnabled' => true]);
            $pdf->loadView('admin.solicitud_de_compras.print_pdf_sol_compra', compact('solicitud_compra', 'orden_compra_solicitud_compra_clon_id'));

            $nombre_pdf = $solicitud_compra->id . '_temp_Solicitud_de_compra.pdf';
            $pdf_sol_get = storage_path('app/public/pdf_solicitud_temp/').$nombre_pdf;
            @unlink($pdf_sol_get);
            $pdf->save($pdf_sol_get);

            if ($enviar_correo == 1){
                if (!empty($solicitud_compra->emails)){
                    $email_asunto = ($solicitud_compra->email_asunto) ? $solicitud_compra->email_asunto : 'Solicitud de Compra';
                    $msg = [
                        'subject' => $email_asunto,
                        'content' => $solicitud_compra->email_contenido,
                        'nombre_codigo' => 'Solicitud de Compra con código',
                    ];
                    $emails_array = explode(',', $solicitud_compra->emails);
                    foreach ($emails_array as $email) {
                        Mail::to($email)->queue(new MessageReceived($solicitud_compra, $pdf_sol_get, $msg, 'emails.message-received', 'Solicitud_de_compra.pdf'));
                    }
                } else {
                    $response = [
                        'status' => "ERROR",
                        'error' => true,
                        'message' => 'No se encontraron emails',
                    ];
                    return response()->json($response);
                }
            } else{
                //return $pdf->stream();
                return $pdf->download('Solicitud_de_compra.pdf');
            }
        } catch (\Exception $e) {
            Log::info('Exception => ' . $e->getMessage() . ' in ' . $e->getLine());
            $response = [
                'status' => "ERROR",
                'error' => true,
                'message' => 'Ocurrio un error al generar el pdf, contacte al administrador del sistema.',
                //'errors' => $e->getMessage() . ' in ' . $e->getLine()
            ];
            return response()->json($response);
        }
    }

    public function eliminar_material_bd($id_material)
    {
        try {
            $resultadoErrors = []; //Lista de Errores

            $material_delete = SolicitudCompraProducto::find($id_material);
            if (!$material_delete->delete()) {
                array_push($resultadoErrors, 'Ocurrio un error al intentar eliminar material');
            } else {
                //elimnar si tiene precios
                $material_precio_delete = RelSolicitudCompraProductoPrecio::where('solicitud_compra_producto_id', $material_delete->id)->delete();

                //Eliminar Comentario grupal
                $comentario_aux = RelSolicitudProductoComentarioMaterial::with('comentariosMaterial')
                ->where('solicitud_compra_producto_id' , $material_delete->id)->first();
                $comentario_aux_exist = $comentario_aux;

                if (!empty($comentario_aux)){
                    $rel_Comentario_Material = RelSolicitudProductoComentarioMaterial::where('solicitud_compra_producto_id' , $material_delete->id)->delete();

                    $id_comentario_aux = $comentario_aux_exist->comentariosMaterial->id;

                    $rel_Comentario_Material_exist = RelSolicitudProductoComentarioMaterial::where('comentarios_materiales_id', $id_comentario_aux)->get();
                    if (count($rel_Comentario_Material_exist) <= 0){
                        $material_delete = ComentarioMaterial::where('id', $id_comentario_aux)->delete();
                    }
                }

            }

            if (!empty($resultadoErrors)) {
                $response = [
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'Ocurrio un error al intentar eliminar material.',
                    'errors' => $resultadoErrors
                ];
            } else {
                $response = [
                    'status' => "OK",
                    'error' => false,
                    'message' => 'Material eliminado correctamente.',
                ];
            }

            return response()->json($response);

        } catch (\Exception $ex) {
            Log::info('Exception: eliminar_material_bd => ' . $ex->getMessage() . ' in ' . $ex->getLine());
            return response()->json([
                'error' => true,
                'message' => $ex->getMessage(),
            ]);
        }
    }

    public function print_pdf_list_selected(Request $request)
    {
        try {
            $dataAll = $request->all();
            $resultadoErrors = []; //Lista de Errores

            $validator = \Validator::make($dataAll[0], [
                'id' => 'required|integer',
            ]);

            if ($validator->fails()) {
                $response = [
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'Parámetros incompletos.',
                ];
            } else {
                $workflowIds = [];
                foreach ($dataAll as $item) {
                    array_push($workflowIds, $item['id']);
                }

                $solicitudes_de_compras = SolicitudCompra::with([
                    'obra',
                    'ingeniero',
                    'estatus',
                    'solicitudCompraProducto',
                    'solicitudCompraProducto.producto',
                    'solicitudCompraProducto.unidad',
                    'solicitudCompraProducto.RelSolicitudProductoComentarioMaterial.comentariosMaterial',
                ])
                    ->whereIn('id', $workflowIds)
                    ->whereNull('deleted_at')
                    ->orderBy('id', 'desc')
                    ->get();

                $carpeta_path = 'solicitudes_de_compras_pdf_listado_temporal/';
                if (env('APP_ENV') == 'production'){
                    #$path_public = '/home/grupoari/appordenes.grupoarista.com.mx/'.$carpeta_path;
                    $path_public = public_path($carpeta_path);
                } else {
                    $path_public = public_path($carpeta_path);
                }

                $orden_compra_solicitud_compra_clon_id = null;
                //generar lista pdf
                $pdf = App::make('dompdf.wrapper');
                $pdf->setOption(['dpi' => 110, 'defaultFont' => 'sans-serif', 'isPhpEnabled' => true]);
                $pdf->loadView('admin.solicitud_de_compras.print_pdf_lista_solicitudes_de_compras', compact('solicitudes_de_compras', 'orden_compra_solicitud_compra_clon_id'));

                $file_name = time() . '_solicitudes_de_compras.pdf';
                //$pdf_sol_get = storage_path('app/public/').$carpeta_path.$file_name;
                //$pdf->save($pdf_sol_get);
                $pdf->save($path_public.$file_name);

//                $pdfMerger = PDFMerger::init(); //Initialize the merger
//                $pdfMerger->addPDF($pdf_sol_get, 'all');
//                foreach ($solicitudes_de_compras as $solicitud_compra) {
//                    if ($solicitud_compra->url_descarga_pdf){
//                        $url_descarga_pdf = storage_path('app/public/').$solicitud_compra->url_descarga_pdf;
//                        $pdfMerger->addPDF($url_descarga_pdf, 'all');
//                    }
//                }
//                $pdfMerger->merge();
//                $pdfMerger->save($path_public.$file_name);

                $archivo_url = $carpeta_path . $file_name;
                //@unlink($pdf_sol_get);

                $response = [
                    'status' => "OK",
                    'error' => false,
                    'message' => 'PDF generado con éxito.',
                    'archivo_url' => $archivo_url,
                ];
            }
            return response()->json($response);

        } catch (\Exception $e) {
            Log::info('Exception => ' . $e->getMessage() . ' in ' . $e->getLine());
            $response = [
                'status' => "ERROR",
                'error' => true,
                'message' => 'Ocurrio un error al generar el pdf, contacte al administrador del sistema.',
            ];
            return response()->json($response);
        }
    }

    public function actualizar_estatus_autorizado(Request $request)
    {
        $response = [];
        $dataAll = $request->all();

        $validator = \Validator::make($dataAll, [
            'id_solicitud' => 'required',
        ]);

        if ($validator->fails()) {
            $response = [
                'status' => "ERROR",
                'error' => true,
                'message' => 'Campos obligatorios',
                'errors' => $validator->errors()->all()
            ];
        } else {
            $solicitud_compra = SolicitudCompra::where('id', $dataAll['id_solicitud'])->first();

            //Estatus: Autorizado
            $autorizado = Estatus::where('clave', 'autorizado')->whereNull('deleted_at')->first();
            if (!empty($autorizado)){
                if ($solicitud_compra->update(['estatus_id' => $autorizado->id])){
                    $response = [
                        'status' => "OK",
                        'error' => false,
                        'message' => 'Solicitud de compra Autorizada con éxito.',
                    ];
                } else {
                    $response = [
                        'status' => "ERROR",
                        'error' => true,
                        'message' => 'No se puede Autorizar la Solicitud de compra.',
                    ];
                }
            } else {
                $response = [
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'No se puede Autorizar la Solicitud de compra.',
                ];
            }
        }
        return response()->json($response);
    }

    public function get_Lista_Materiales_By_Solicitud_Id_Select(Request $request, $solicitud_id)
    {
        try {
            $lista_materiales_solicitud = SolicitudCompraProducto::with('producto')
                ->whereNull('deleted_at')
                ->where('solicitud_compra_id', $solicitud_id)
                ->get();

            if(!$lista_materiales_solicitud->isEmpty()) {
                $valid_tags = [];
                foreach ($lista_materiales_solicitud as $item) {
                    if (!$request->has('type')) {
                        $lista_libres = RelSolicitudProductoComentarioMaterial::where('solicitud_compra_producto_id', $item->id)->first();
                        if (empty($lista_libres)) {
                            $valid_tags[] = ['id' => $item->id, 'text' => $item->producto->nombre];
                        }
                    } else {
                        //$type = $request->type;
                        $valid_tags[] = ['id' => $item->id, 'text' => $item->producto->nombre];
                    }
                }
                return response()->json($valid_tags);
            }else{
                return response()->json([
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'No se encontraron resultados'
                ]);
            }

        } catch (\Exception $ex) {
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'message' => 'No se encontraron resultados',
                'message_aux' => $ex->getMessage(),
            ]);
        }
    }

    /**
     * Guarda un comentario para un grupo de Materiales
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function saveComentarioGrupal(Request $request)
    {
        $response = [];
        $dataAll = $request->all();
        $resultadoErrors = []; //Lista de Errores

        $validator = \Validator::make($dataAll, [
            'solicitud_compra_id' => 'required|integer',
            'clave_solicitud_compra_producto' => 'required|array|min:1',
            'comentario' => 'required',
        ]);

        if ($validator->fails()) {
            $response = [
                'status' => "ERROR",
                'error' => true,
                'message' => 'Campos obligatorios',
                'errors' => $validator->errors()->all()
            ];
        } else {
            $data_success = [];

            $comentario = new ComentarioMaterial();
            $comentario->solicitud_compra_id = $dataAll['solicitud_compra_id'];
            $comentario->comentario = $dataAll['comentario'];
            $comentario->solicitud_compra_productos_ids = implode(",", $dataAll['clave_solicitud_compra_producto']);
            if ($comentario->save()) {
                array_push($data_success, $comentario);

                //lista solicitud compra producto
                $clave_data = $dataAll['clave_solicitud_compra_producto'];
                foreach ($clave_data as $clave) {
                    $comentarioGrupal = new RelSolicitudProductoComentarioMaterial();

                    $comentarioGrupal->solicitud_compra_producto_id = $clave;
                    $comentarioGrupal->comentarios_materiales_id = $comentario->id;
                    $comentarioGrupal->users_id = Auth::id();
                    $comentarioGrupal->solicitud_compra_id = $dataAll['solicitud_compra_id'];

                    if (!$comentarioGrupal->save()) {
                        array_push($resultadoErrors, 'Ocurrio un error al intentar guardar comentario material');
                    } else {
                        array_push($data_success, $comentarioGrupal);
                    }
                }

                if (!empty($resultadoErrors)) {
                    $response = [
                        'status' => "ERROR",
                        'error' => true,
                        'message' => 'Ocurrio un error al guardar Comentario para el/los Material.',
                        'errors' => $resultadoErrors
                    ];
                } else {
                    $response = [
                        'status' => "OK",
                        'error' => false,
                        'message' => 'Comentario Material guardado correctamente.',
                        'data' => $data_success
                    ];
                }

            } else {
                $response = [
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'Ocurrio un error al guardar el comentario.',
                    'errors' => 'New ComentarioMaterial: error al guardar el comentario.'
                ];
            }

        }

        return response()->json($response);
    }

    public function getComentarioGrupalById($id_comentario_material)
    {
        try {
            $comentarioGrupal = ComentarioMaterial::find($id_comentario_material);

            $materiales = RelSolicitudProductoComentarioMaterial::with(['comentariosMaterial', 'solicitudCompraProducto', 'solicitudCompraProducto.producto'])
                ->where('comentarios_materiales_id', $id_comentario_material)->get();

            return response()->json([
                'error' => false,
                'mensajes' => 'Material encontrado',
                'comentario_material' => $comentarioGrupal,
                'materiales' => $materiales
            ]);

        }catch (\Exception $ex) {
            return response()->json([
                'error' => true,
                'mensajes' => $ex->getMessage(),
            ]);
        }
    }

    /**
     * Actualiza un comentario para un grupo de Materiales
     * @param Request $request
     * @param $comentario_material_id
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateComentarioGrupal(Request $request, $comentario_material_id)
    {
        try {
            $response = [];
            $dataAll = $request->all();
            $resultadoErrors = []; //Lista de Errores

            $validator = \Validator::make($dataAll, [
                'comentario_material_clave_edit' => 'required',
                'comentario_edit' => 'required',
                'solicitud_compra_id' => 'required|integer',
            ]);

            if ($validator->fails()) {
                $response = [
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'Campos obligatorios',
                    'errors' => $validator->errors()->all()
                ];
            } else {
                $comentarioGrupal = ComentarioMaterial::find($comentario_material_id);
                //actualiza comentario
                $comentarioGrupal->comentario = $dataAll['comentario_edit'];
                if ($comentarioGrupal->update()) {

                    //$materiales_no_delete_ids[];
                    $materiales_no_delete_ids_update = [];
                    if (count($dataAll['claves_solicitud_compra_producto_edit'])) {
                        foreach ($dataAll['claves_solicitud_compra_producto_edit'] as $item_material_edit) {
                            // Guardar el ID del material procesado
                            $materiales_no_delete_ids[] = $item_material_edit;

                            //obtener las claves
                            $materiales_update_claves = RelSolicitudProductoComentarioMaterial::find($item_material_edit);
                            $materiales_no_delete_ids_update[] = $materiales_update_claves->solicitud_compra_producto_id;

                        }
                    }

                    if (count($materiales_no_delete_ids_update) > 0){
                        $comentarioGrupal->solicitud_compra_productos_ids = implode(",", $materiales_no_delete_ids_update);
                        $comentarioGrupal->update();
                    }

                    //Eliminar materiales rel comentario grupal
                    if (!empty($materiales_no_delete_ids) && count($materiales_no_delete_ids) > 0) {
                        $materiales_rel_cometario_delete = RelSolicitudProductoComentarioMaterial::whereNotIn('id', $materiales_no_delete_ids)
                            ->where('comentarios_materiales_id', $comentario_material_id)->delete();
                    } else {
                        $materiales_rel_cometario_delete = RelSolicitudProductoComentarioMaterial::where('comentarios_materiales_id', $comentario_material_id)
                            ->delete();
                    }

                    //nuevo material rel comentario grupal
                    if (count($dataAll['claves_nuevas_materiales_solicitud']) > 0) {
                        $productos_ids = array_merge($materiales_no_delete_ids_update, $dataAll['claves_nuevas_materiales_solicitud']);
                        $comentarioGrupal->solicitud_compra_productos_ids = implode(",", $productos_ids);
                        $comentarioGrupal->update();

                        foreach ($dataAll['claves_nuevas_materiales_solicitud'] as $item_agrega_material) {
                            $comentario_material = new RelSolicitudProductoComentarioMaterial();
                            $comentario_material->solicitud_compra_producto_id = $item_agrega_material;
                            $comentario_material->comentarios_materiales_id = $comentario_material_id;
                            $comentario_material->users_id = Auth::id();
                            $comentario_material->solicitud_compra_id = $dataAll['solicitud_compra_id'];

                            if (!$comentario_material->save()) {
                                array_push($resultadoErrors, 'Ocurrio un error al intentar guardar comentario material');
                            }
                        }
                    }

                    if (empty($materiales_no_delete_ids) && count($dataAll['claves_nuevas_materiales_solicitud']) <= 0) {
                        ComentarioMaterial::find($comentario_material_id)->delete();
                    }

                    if (!empty($resultadoErrors)) {
                        $response = [
                            'status' => "ERROR",
                            'error' => true,
                            'message' => 'Ocurrio un error al actualizar Comentario para el/los Material.',
                            'errors' => $resultadoErrors
                        ];
                    } else {
                        $response = [
                            'status' => "OK",
                            'error' => false,
                            'message' => 'Comentario Material actualizado correctamente.',
                        ];
                    }

                } else {
                    $response = [
                        'status' => "ERROR",
                        'error' => true,
                        'message' => 'Ocurrio un error al actualizar el comentario comentario grupal.',
                        'errors' => 'New ComentarioMaterial: error al guardar el comentario.'
                    ];
                }
            }

            return response()->json($response);

        } catch (\Exception $ex){
            DB::rollBack();
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'message' => $ex->getMessage(),
            ]);
        }
    }

    /**
     * Estatus Finalizado para cerrar la S.C.
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function cambiarEstatusFinalizado(Request $request)
    {
        $response = [];
        $dataAll = $request->all();

        $validator = \Validator::make($dataAll, [
            'id_solicitud' => 'required',
        ]);

        if ($validator->fails()) {
            $response = [
                'status' => "ERROR",
                'error' => true,
                'message' => 'Campos obligatorios',
                'errors' => $validator->errors()->all()
            ];
        } else {
            $solicitud_compra = SolicitudCompra::where('id', $dataAll['id_solicitud'])->first();

            //Estatus: Finalizado
            $finalizado = Estatus::where('clave', 'finalizado')->whereNull('deleted_at')->first();
            if (!empty($finalizado)){
                if ($solicitud_compra->update(['estatus_id' => $finalizado->id])) {
                    $response = [
                        'status' => "OK",
                        'error' => false,
                        'message' => 'Solicitud de compra Finalizada con éxito.',
                    ];
                } else {
                    $response = [
                        'status' => "ERROR",
                        'error' => true,
                        'message' => 'No se puede Finalizar la Solicitud de compra.',
                    ];
                }
            } else {
                $response = [
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'No se puede Finalizar la Solicitud de compra.',
                ];
            }
        }
        return response()->json($response);
    }

    /**
     * Crear una replica exacta de la S.C.
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function saveClonSolicitud(Request $request)
    {
        try {
            $response = [];
            $dataAll = $request->all();
            $resultadoErrors = []; //Lista de Errores

            $validator = \Validator::make($dataAll, [
                'solicitud_compra_id' => 'required',
            ]);

            if ($validator->fails()) {
                array_push($resultadoErrors, 'Campos obligatorios: ' . implode(',', $validator->errors()->all()));
            } else {
                DB::beginTransaction();

                $solicitud_compra = SolicitudCompra::with([
                    'solicitudCompraProducto',
                    'solicitudCompraProducto.RelSolicitudProductoComentarioMaterial.comentariosMaterial',
                ])->where('id', $dataAll['solicitud_compra_id'])->first();

                $estatus_por_autorizar = Estatus::where('clave', 'por_autorizar')->whereNull('deleted_at')->first();

                $clon_solicitud = new SolicitudCompra();
                $clon_solicitud->obra_id = $solicitud_compra->obra_id;
                $clon_solicitud->ingeniero_id = $solicitud_compra->ingeniero_id;
                $clon_solicitud->observaciones = $solicitud_compra->observaciones;
                $clon_solicitud->fecha_solicitud = Carbon::now()->format('Y-m-d');
                $solicitud_compra->fecha_entrega = (Carbon::createFromFormat('d/m/Y', $dataAll['fecha_entrega']))->format('Y-m-d');
                $clon_solicitud->estatus_id = $estatus_por_autorizar->id;
                $clon_solicitud->emails = $solicitud_compra->emails;
                $clon_solicitud->email_asunto = $solicitud_compra->email_asunto;
                $clon_solicitud->email_contenido = $solicitud_compra->email_contenido;
                $clon_solicitud->clave_version = Consts::SOLICITUD_COMPRA_V2;

                if ($clon_solicitud->save()) {
                    $foliador = $this->getFoliador($clon_solicitud->id);
                    $clon_solicitud->codigo = $foliador;
                    $clon_solicitud->codigo_nombre = 'solicitud-de-compra-'.$foliador;
                    $clon_solicitud->update();

                    //lista materiales
                    foreach ($solicitud_compra->solicitudCompraProducto as $producto) {
                        $solicitud_compra_producto = new SolicitudCompraProducto();
                        $solicitud_compra_producto->solicitud_compra_id = $clon_solicitud->id;
                        $solicitud_compra_producto->producto_id = $producto->producto_id;
                        $solicitud_compra_producto->unidad_id = $producto->unidad_id;
                        $solicitud_compra_producto->cantidad = $producto->cantidad;
                        $solicitud_compra_producto->area_de_uso = $producto->area_de_uso;
                        $solicitud_compra_producto->marca = $producto->marca;
                        $solicitud_compra_producto->modelo = $producto->modelo;
                        $solicitud_compra_producto->observaciones = $producto->observaciones;

                        if ($solicitud_compra_producto->save()) {
                            //Existe Comentario Grupal
                            if (!empty($producto->RelSolicitudProductoComentarioMaterial)) {
                                $comentario = new ComentarioMaterial();
                                $comentario->comentario = $producto->RelSolicitudProductoComentarioMaterial->comentariosMaterial->comentario;
                                if ($comentario->save()) {
                                    $comentarioGrupal = new RelSolicitudProductoComentarioMaterial();
                                    $comentarioGrupal->solicitud_compra_producto_id = $solicitud_compra_producto->id;
                                    $comentarioGrupal->comentarios_materiales_id = $comentario->id;
                                    $comentarioGrupal->users_id = Auth::id();
                                    if (!$comentarioGrupal->save()) {
                                        array_push($resultadoErrors, 'Ocurrio un error al intentar guardar comentario material.');
                                    }
                                } else {
                                    array_push($resultadoErrors, 'Ocurrio un error al intentar guardar comentario.');
                                }
                            }
                        } else {
                            array_push($resultadoErrors, 'Ocurrio un error al intentar agregar material');
                        }
                    }

                } else {
                    array_push($resultadoErrors, 'Ocurrio un error al clonar la solicitud de compra.');
                }
            }

            if (empty($resultadoErrors)) {
                DB::commit();
                $response = [
                    'status' => "OK",
                    'error' => false,
                    'message' => 'La solicitud de compra se ha clonado con éxito.',
                ];
            } else {
                DB::rollBack();
                $response = [
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'Ocurrio un error al clonar la solicitud de compra.',
                    'errors' => $resultadoErrors
                ];
            }

            return response()->json($response);

        } catch (\Exception $ex){
            DB::rollBack();
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'mensajes' => $ex->getMessage(),
            ]);
        }
    }

}
