<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Producto;
use Illuminate\Http\Request;
use DataTables;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class ProductosController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();

        $catalogo_de_productos_edit = $user->can('catalogo_de_productos_edit');
        $catalogo_de_productos_destroy = $user->can('catalogo_de_productos_destroy');

        return view('admin.catalogos.productos.lista', compact('catalogo_de_productos_edit', 'catalogo_de_productos_destroy'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('admin.catalogos.productos.nuevo');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'nombre' => 'required|min:3',
        ]);

        $producto = new Producto();
        $producto->nombre = $request->input('nombre');
        $producto->descripcion = $request->input('descripcion');

        if ($producto->save()){
            $foliador = $this->getFoliador($producto->id);
            $producto->codigo = $foliador;
            $producto->update();
        }

        return redirect()->route('catalogo-de-productos.index')->withSuccess('Producto guardado con éxito.');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $producto = Producto::findOrFail($id);
        return view('admin.catalogos.productos.editar', compact("producto"));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validated = $request->validate([
            'nombre' => 'required|min:3',
        ]);

        $producto = Producto::findOrFail($id);
        $producto->nombre = $request->input('nombre');
        $producto->descripcion = $request->input('descripcion');
        $producto->update();

        return redirect()->route('catalogo-de-productos.index')->withSuccess('Producto actualizado con éxito.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $producto = Producto::destroy($id);

            return response()->json(['error' => false, 'message' => 'Registro eliminado.']);
        } catch (\Exception $ex) {
            return response()->json([
                'error' => true,
                'message' => $ex->getMessage(),
            ]);
        }
    }

    public function get_Lista_Productos(Request $request)
    {
        if ($request->ajax()) {
            $productos = Producto::select('id', 'nombre', 'codigo', 'descripcion')
                ->whereNull('deleted_at')
                ->orderBy('id', 'desc')->get();

            return DataTables::of($productos)->make(true);
        }
    }
    public function get_Lista_Productos_Select(Request $request)
    {
        $productos = [];
        if($request->has('search')){
            $search = $request->search;

            $productos = Producto::select('id', 'nombre', 'codigo')
                ->whereNull('deleted_at')
                ->where('nombre', 'LIKE', '%'.$search.'%')
                ->orderBy('nombre', 'asc')
                ->get();
        }else{
            $productos = Producto::select('id', 'nombre', 'codigo')
                ->whereNull('deleted_at')
                ->orderBy('nombre', 'asc')
                ->get();
        }

        if(!$productos->isEmpty()){
            $valid_tags = [];
            foreach ($productos as $item) {
                $valid_tags[] = ['id' => $item->id, 'text' => $item->nombre];
            }
            return response()->json($valid_tags);
        }else{
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'message' => 'No se encontraron resultados'
            ]);
        }

    }

    public function getProductoById($id)
    {
        try {
            $producto = Producto::findOrFail($id);
            return response()->json([
                'error' => false,
                'mensajes' => 'Material encontrado',
                'producto' => $producto
            ]);
        }catch(\Exception $ex){
            return response()->json([
                'error' => true,
                'mensajes' => $ex->getMessage() . ' in line ' . $ex->getLine(),
            ]);
        }
    }

    private function getFoliador($ultimo_id = 0)
    {
        $foliador = 1;
//        $ultimo_id = Producto::latest('id')->first();
        if ($ultimo_id > 0){
            $foliador = str_pad($ultimo_id,  4, "0", STR_PAD_LEFT);
        }else{
            $foliador = str_pad($foliador,  4, "0", STR_PAD_LEFT);
        }
        return $foliador;
    }

    public function crearMaterial(Request $request)
    {
        $response = [];
        $dataAll = $request->all();
        $cError = ""; //Errores
        $resultadoErrors = []; //Lista de Errores

        $validator = \Validator::make($dataAll, ['nombre' => 'required|min:3']);

        if ($validator->fails()) {
            $response = [
                'status' => "ERROR",
                'error' => true,
                'message' => 'Campos obligatorios',
                'errors' => $validator->errors()->all()
            ];
        } else {
            $producto = new Producto();
            $producto->nombre = $dataAll['nombre'];
            $producto->descripcion = $dataAll['descripcion'];

            if ($producto->save()){
                $foliador = $this->getFoliador($producto->id);
                $producto->codigo = $foliador;
                $producto->update();

                $response = [
                    'status' => "OK",
                    'error' => false,
                    'message' => 'Material guardado con éxito.',
                    'data' => $producto
                ];
            } else {
                $response = [
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'Ocurrio un error al guardar el Material.',
                    'errors' => $resultadoErrors
                ];
            }
        }

        return response()->json($response);
    }

}
