<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Mail\MessageReceived;
use App\Models\Estatus;
use App\Models\OrdenCompra;
use App\Models\PagoProveedor;
use Carbon\Carbon;
use Illuminate\Http\Request;
use DataTables;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use PDFSnappy;
use LynX39\LaraPdfMerger\Facades\PdfMerger;

class PagosProveedoresController extends Controller
{
    protected $path_archivo_pdf;
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('can:pagos_a_proveedores_index')->only('index');
        $this->middleware('can:pagos_a_proveedores_create')->only('create', 'store');
        $this->middleware('can:pagos_a_proveedores_edit')->only('edit', 'update');
        $this->middleware('can:pagos_a_proveedores_destroy')->only('destroy');
        $this->middleware('can:pagos_a_proveedores_show')->only('show');
        $this->path_archivo_pdf = ('pagos_proveedores_pdf');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();

        $pagos_a_proveedores_show = $user->can('pagos_a_proveedores_show');
        $pagos_a_proveedores_edit = $user->can('pagos_a_proveedores_edit');
        $pagos_a_proveedores_destroy = $user->can('pagos_a_proveedores_destroy');
        $pagos_a_proveedores_export  = $user->can('pagos_a_proveedores_export');
        $pagos_a_proveedores_bulk_upload = $user->can('pagos_a_proveedores_bulk_upload');
        $pagos_a_proveedores_send_email = $user->can('pagos_a_proveedores_send_email');
        $pagos_a_proveedores_authorized  = $user->can('pagos_a_proveedores_authorized');
        $pagos_a_proveedores_canceled  = $user->can('pagos_a_proveedores_canceled');

        //Estatus: Por autorizar
        $estatus_por_autorizar = Estatus::where('clave', 'por_autorizar')->whereNull('deleted_at')->first();
        //Estatus: Autorizado
        $estatus_autorizado = Estatus::where('clave', 'autorizado')->whereNull('deleted_at')->first();
        //Estatus: Cancelado
        $estatus_cancelado = Estatus::where('clave', 'cancelado')->whereNull('deleted_at')->first();

        return view('admin.pagos_a_proveedores.lista', compact(
            'pagos_a_proveedores_show',
            'pagos_a_proveedores_edit',
            'pagos_a_proveedores_destroy',
            'pagos_a_proveedores_export',
            'pagos_a_proveedores_bulk_upload',
            'pagos_a_proveedores_send_email',
            'pagos_a_proveedores_authorized',
            'pagos_a_proveedores_canceled',
            'estatus_por_autorizar',
            'estatus_autorizado',
            'estatus_cancelado',
        ));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('admin.pagos_a_proveedores.nuevo');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            $response = [];
            $dataAll = $request->all();
            $resultadoErrors = []; //Lista de Errores

            $validator = \Validator::make($dataAll, [
                'orden_compra_id' => 'required|integer',
                'fecha_pago' => 'required',
                'conceptos_materiales' => 'required',
                'archivo' => 'mimes:pdf',
            ]);

            if ($validator->fails()) {
                array_push($resultadoErrors, 'Campos obligatorios, msg: ' . $validator->errors()->all());
            } else {
                DB::beginTransaction();

                $orden_compra = OrdenCompra::with([
                    'proveedor',
                    'ivaImpuesto',
                    'estatus',
                    'ordenCompraAnticipo',
                    'solicitudCompra',
                    'solicitudCompra.obra',
                    'solicitudCompra.ingeniero',
                    'solicitudCompra.solicitudCompraProducto',
                    'solicitudCompra.solicitudCompraProducto.producto',
                    'solicitudCompra.solicitudCompraProducto.unidad',
                ])
                    ->where('id', $dataAll['orden_compra_id'])
                    ->where('estatus_id', 3)
                    ->where('enviado_a_pago', 0)
                    ->whereNull('deleted_at')
                    ->first();

                if (!empty($orden_compra)) {

                    //Estatus Por autorizar
                    $estatus_por_autorizar = Estatus::where('clave', 'por_autorizar')->whereNull('deleted_at')->first();

                    $pago_proveedor = new PagoProveedor();
                    $pago_proveedor->orden_compra_id = $orden_compra->id;
                    $pago_proveedor->proveedor_id = $orden_compra->proveedor_id;
                    $pago_proveedor->obra_id = $orden_compra->solicitudCompra->obra_id;
                    $pago_proveedor->fecha_pago = $this->getFechaPago($dataAll['fecha_pago']);
                    $pago_proveedor->conceptos_materiales = $dataAll['conceptos_materiales'];
                    $pago_proveedor->comentarios = $dataAll['comentarios'];
                    $pago_proveedor->estatus_id = $estatus_por_autorizar->id; //Por autorizar
                    $pago_proveedor->importe = $orden_compra->total;
                    $pago_proveedor->importe_letras = $orden_compra->total_letras;
                    $pago_proveedor->url_factura_pdf = $request->input('url_factura_pdf');
                    $pago_proveedor->url_resumen_pdf = $request->input('url_resumen_pdf');
                    $pago_proveedor->emails = $dataAll['emails'];
                    $pago_proveedor->email_asunto = $dataAll['email_asunto'];
                    $pago_proveedor->email_contenido = $dataAll['email_contenido'];

                    if ($pago_proveedor->save()) {
                        $foliador = $this->getFoliador($pago_proveedor->id);
                        $pago_proveedor->update(['codigo' => $foliador]);

                        $orden_compra->update(['enviado_a_pago' => 1]); // Enviado a pago de proveedor

                    } else {
                        array_push($resultadoErrors, 'No se pudó guardar el pago a proveedor.');
                    }
                } else {
                    array_push($resultadoErrors, 'No se encontró Orden de compra');
                }

                if (empty($resultadoErrors)) {
                    DB::commit();
                    $response = [
                        'status' => "OK",
                        'error' => false,
                        'message' => 'Pago a proveedor guardado con éxito.',
                        'errors' => ''
                    ];
                } else {
                    DB::rollBack();
                    $response = [
                        'status' => "ERROR",
                        'error' => true,
                        'message' => 'Ocurrio un error al guardar el Pago a proveedor.',
                        'errors' => $resultadoErrors
                    ];
                }
            }

            return response()->json($response);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::info('Exception => ' . $e->getMessage() . ' in ' . $e->getLine());
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'message' => 'Ocurrio un error al guardar el Pago a proveedor, contacte al administrador del sistema.',
            ]);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $pago_proveedor = PagoProveedor::with(['ordenCompra', 'proveedor', 'obra', 'estatus'])
            ->where('id', $id)
            ->whereNull('deleted_at')
            ->first();

        return view('admin.pagos_a_proveedores.info', compact('pago_proveedor'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $pago_proveedor = PagoProveedor::with(['ordenCompra', 'proveedor', 'obra', 'estatus'])
            ->where('id', $id)
            ->whereNull('deleted_at')
            ->first();

        return view('admin.pagos_a_proveedores.editar', compact('pago_proveedor'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        try {
            $response = [];
            $dataAll = $request->all();
            $resultadoErrors = []; //Lista de Errores

            $validator = \Validator::make($dataAll, [
                //'orden_compra_id' => 'required|integer',
                'fecha_pago' => 'required',
                'conceptos_materiales' => 'required',
                //'archivo' => 'mimes:pdf',
            ]);

            if ($validator->fails()) {
                array_push($resultadoErrors, 'Campos obligatorios, msg: ' . $validator->errors()->all());
            } else {
                DB::beginTransaction();

                $pago_proveedor = PagoProveedor::find($id);
                $pago_proveedor->fecha_pago = $this->getFechaPago($dataAll['fecha_pago']);
                $pago_proveedor->conceptos_materiales = $dataAll['conceptos_materiales'];
                $pago_proveedor->comentarios = $dataAll['comentarios'];
                $pago_proveedor->url_factura_pdf = $dataAll['url_factura_pdf'];
                $pago_proveedor->url_resumen_pdf = $dataAll['url_resumen_pdf'];
                $pago_proveedor->emails = $dataAll['emails'];
                $pago_proveedor->email_asunto = $dataAll['email_asunto'];
                $pago_proveedor->email_contenido = $dataAll['email_contenido'];


                if (!$pago_proveedor->update()) {
                    array_push($resultadoErrors, 'No se pudó actualizar el pago a proveedor.');
                }

                if (empty($resultadoErrors)) {
                    DB::commit();
                    $response = [
                        'status' => "OK",
                        'error' => false,
                        'message' => 'Pago a proveedor actualizado con éxito.',
                        'errors' => ''
                    ];
                } else {
                    DB::rollBack();
                    $response = [
                        'status' => "ERROR",
                        'error' => true,
                        'message' => 'Ocurrió un error al actualizar el Pago a proveedor.',
                        'errors' => $resultadoErrors
                    ];
                }
            }

            return response()->json($response);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::info('Exception => ' . $e->getMessage() . ' in ' . $e->getLine());
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'message' => 'Ocurrio un error al actualizar el Pago a proveedor, contacte al administrador del sistema.',
            ]);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $pago_proveedor = PagoProveedor::with('ordenCompra')->findOrFail($id);
        if ($pago_proveedor->delete()) {
            $pago_proveedor->ordenCompra->update(['enviado_a_pago' => 0]);
        }
        return response()->json(['error' => false, 'message' => 'Registro eliminado.']);
    }

    public function get_Lista_Pagos_Proveedores(Request $request)
    {
        if ($request->ajax()) {
            $pagos_proveedores = PagoProveedor::with([
                'ordenCompra',
                'ordenCompra.ordenCompraAnticipo',
                'proveedor', 'obra', 'estatus'
                ])
                ->whereNull('deleted_at')
                ->orderBy('id', 'desc')->get();

            $pagos_proveedores->map(function ($pagoProveedor) {
                $pagoProveedor->importe = number_format($pagoProveedor->importe, 2);

                //Lista de Anticipos
                $suma_array_anticipos = array();
                $lista_anticipos = $pagoProveedor->ordenCompra->ordenCompraAnticipo;
                if (!empty($lista_anticipos)) {
                    foreach ($lista_anticipos as $key => $anticipo) {
                        $suma_array_anticipos[$key] = floatval($anticipo->anticipo);
                        $anticipo->tipo_pago = $pagoProveedor->ordenCompra->tipoPago;

                        if ($key !== count($lista_anticipos)-1) {
                            $anticipo->es_ultimo = false;
                        } else {
                            $anticipo->es_ultimo = true;
                        }
                    }
                }
                $total_sum_anticipos = array_sum($suma_array_anticipos);
                $pagoProveedor->total_sum_anticipos = number_format($total_sum_anticipos, 2);
            });

            return DataTables::of($pagos_proveedores)->make(true);
        }
    }

    public function delete_Archivo_Pago_Pdf($id)
    {
        $pago_proveedor = PagoProveedor::findOrFail($id);
        $name_pdf = $pago_proveedor->url_archivo_pdf;
        $pago_proveedor->url_archivo_pdf = '';
        $pago_proveedor->update();

        @unlink($this->path_archivo_pdf . '/' . $name_pdf);

        return redirect('/pagos-a-proveedores/' . $id . '/edit')->withSuccess('Archivo PDF eliminado.');
    }

    private function getFoliador($ultimo_id = 0)
    {
        $foliador = 1;
        if ($ultimo_id > 0) {
            $foliador = str_pad($ultimo_id,  4, "0", STR_PAD_LEFT);
        } else {
            $foliador = str_pad($foliador,  4, "0", STR_PAD_LEFT);
        }
        return $foliador;
    }

    private function getFechaPago($fecha_pago)
    {
        $fechaPago_ = Carbon::now()->format('Y-m-d');

        if ($fecha_pago) {
            $fechaPago_ =  (Carbon::createFromFormat('d/m/Y', $fecha_pago))->format('Y-m-d');
        }

        return $fechaPago_;
    }

    public function print_pdf($id, $enviar_correo = 0)
    {
        try {
            $pago_proveedor = PagoProveedor::with(['ordenCompra', 'proveedor', 'obra', 'estatus'])
                ->where('id', $id)
                ->whereNull('deleted_at')
                ->first();

//            $pdf = PDFSnappy::loadView('admin.pagos_a_proveedores.print_pdf', compact('pago_proveedor'));
//            $pdf->setOption('header-spacing', 1);
//            $pdf->setOption('header-html', view('admin.prints.prints_header'));
//            $pdf->setOption('footer-html', view('admin.prints.prints_footer'));
//            $pdf->setOptions([
//                'margin-top'    => 15,
//                'margin-bottom' => 15,
//                'margin-right'  => 15,
//                'margin-left'   => 15,
//            ]);

            $pdf = App::make('dompdf.wrapper');
            $pdf->setOption(['dpi' => 110, 'defaultFont' => 'sans-serif', 'isPhpEnabled' => true]);
            $pdf->loadView('admin.pagos_a_proveedores.print_pdf_pago_prov', compact('pago_proveedor'));

            $nombre_pdf = $pago_proveedor->id . '_temp_Pago_a_proveedor.pdf';
            $pdf_pp_get = storage_path('app/public/pdf_pp_temp/').$nombre_pdf;
            @unlink($pdf_pp_get);
            $pdf->save($pdf_pp_get);

            $pdfMerger = PDFMerger::init(); //Initialize the merger
            $pdfMerger->addPDF($pdf_pp_get, 'all');
            if ($pago_proveedor->url_factura_pdf){
                $url_factura_pdf = storage_path('app/public/').$pago_proveedor->url_factura_pdf;
                $pdfMerger->addPDF($url_factura_pdf, 'all');
            }
            if ($pago_proveedor->url_resumen_pdf){
                $url_resumen_pdf = storage_path('app/public/').$pago_proveedor->url_resumen_pdf;
                $pdfMerger->addPDF($url_resumen_pdf, 'all');
            }
            $pdfMerger->merge();
            $pdfMerger->save($pdf_pp_get);

            if ($enviar_correo == 1) {
                if (!empty($pago_proveedor->emails)) {
                    $email_asunto = ($pago_proveedor->email_asunto) ? $pago_proveedor->email_asunto : 'Pago a proveedor';
                    $msg = [
                        'subject' => $email_asunto,
                        'content' => $pago_proveedor->email_contenido,
                        'nombre_codigo' => 'Pago a proveedor con código',
                    ];
                    $emails_array = explode(',', $pago_proveedor->emails);
                    foreach ($emails_array as $email) {
                        Mail::to($email)->queue(new MessageReceived($pago_proveedor, $pdf_pp_get, $msg, 'emails.message-received', 'Pago_a_proveedor.pdf'));
                    }
                } else {
                    $response = [
                        'status' => "ERROR",
                        'error' => true,
                        'message' => 'No se encontraron emails',
                    ];
                    return response()->json($response);
                }
            } else {
                return $pdfMerger->save($nombre_pdf, "download");
                //return $pdf->download('Pago_a_proveedor.pdf');
            }
            //return $pdf->stream();
        } catch (\Exception $e){
            Log::info('Exception => '. $e->getMessage() . ' in ' . $e->getLine());
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'message' => 'Ocurrio un error al generar el pdf, contacte al administrador del sistema.',
                //'errors' => $e->getMessage() . ' in ' . $e->getLine()
            ]);
        }
    }

    public function print_pdf_all_list()
    {
        try {
            $pagos_proveedores = PagoProveedor::with(['ordenCompra', 'proveedor', 'obra', 'estatus'])
                ->whereNull('deleted_at')
                ->orderBy('id', 'desc')->get();

            $pdf = PDFSnappy::loadView('admin.pagos_a_proveedores.print_pdf_all_list', compact('pagos_proveedores'));
            $pdf->setOption('header-spacing', 1);
            $pdf->setOption('header-html', view('admin.prints.prints_header'));
            $pdf->setOption('footer-html', view('admin.prints.prints_footer'));
            $pdf->setOptions([
                'margin-top'    => 15,
                'margin-bottom' => 15,
                'margin-right'  => 15,
                'margin-left'   => 15,
            ]);

            return $pdf->stream();
        } catch (\Exception $e) {
            Log::info('Exception => ' . $e->getMessage() . ' in ' . $e->getLine());
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'message' => 'Ocurrio un error al generar el pdf, contacte al administrador del sistema.',
                //'errors' => $e->getMessage() . ' in ' . $e->getLine()
            ]);
        }
    }

    public function actualizar_estatus(Request $request)
    {
        try {
            DB::beginTransaction();
            $response = [];
            $dataAll = $request->all();
            $resultadoErrors = []; //Lista de Errores

            $validator = \Validator::make($dataAll, [
                'id_pago' => 'required',
                'estatus_id' => 'required',
            ]);

            if ($validator->fails()) {
                array_push($resultadoErrors, 'Campos obligatorios, msg: ' . $validator->errors()->all());
            } else {

                $pago_proveedor = PagoProveedor::with(['ordenCompra', 'proveedor', 'obra', 'estatus'])
                    ->where('id', $dataAll['id_pago'])
                    ->whereNull('deleted_at')
                    ->first();

                $pago_proveedor->update(['estatus_id' => $dataAll['estatus_id']]);
            }

            if (empty($resultadoErrors)) {
                DB::commit();
                $response = [
                    'status' => "OK",
                    'error' => false,
                    'message' => 'Estatus actualizado correctamente.',
                    'errors' => ''
                ];
            } else {
                DB::rollBack();
                $response = [
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'Ocurrio un error al actualizar el estatus del pago a proveedor.',
                    'errors' => $resultadoErrors
                ];
            }

            return response()->json($response);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::info('Exception => ' . $e->getMessage() . ' in ' . $e->getLine());
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'message' => 'Ocurrio un error al actualizar el estatus, contacte al administrador del sistema.',
                'errors' => "ERROR => " . $e->getMessage() . ' in ' . $e->getLine()
            ]);
        }
    }

    public function uploadPdf(Request $request, $type)
    {
        $request->validate([
            'file' => 'required|mimes:pdf|max:4096',
        ]);

        $file = $request->file('file');
        $fileName = $type . '_' . time() . '_' . Str::uuid() . '.' . $request->file('file')->getClientOriginalExtension();
        $filePath = $request->file('file')->storeAs('pagos_proveedores_pdf', $fileName, 'public');
        $filePath = $this->path_archivo_pdf .'/'. $fileName;
        $file->move($this->path_archivo_pdf, $fileName);

        return response()->json(['filePath' => $filePath]);
    }

    public function print_pdf_list_selected(Request $request)
    {
        try {
            $dataAll = $request->all();
            $resultadoErrors = []; //Lista de Errores

            $validator = \Validator::make($dataAll[0], [
                'id' => 'required|integer',
            ]);

            if ($validator->fails()) {
                $response = [
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'Parámetros incompletos.',
                ];
            } else {
                $workflowIds = [];
                foreach ($dataAll as $item) {
                    array_push($workflowIds, $item['id']);
                }

                $pagos_a_proveedores = PagoProveedor::with(['ordenCompra', 'proveedor', 'obra', 'estatus'])
                    ->whereIn('id', $workflowIds)
                    ->whereNull('deleted_at')
                    ->orderBy('id', 'desc')
                    ->get();

                if (env('APP_ENV') == 'production'){
                    #$path_public = '/home/grupoari/appordenes.grupoarista.com.mx/pago_a_proveedor_pdf_listado_temporal/';
                    $path_public = public_path('pago_a_proveedor_pdf_listado_temporal/');
                } else {
                    $path_public = public_path('pago_a_proveedor_pdf_listado_temporal/');
                }
                //generar lista pdf
                $pdf = App::make('dompdf.wrapper');
                $pdf->setOption(['dpi' => 110, 'defaultFont' => 'sans-serif', 'isPhpEnabled' => true]);
                $pdf->loadView('admin.pagos_a_proveedores.print_pdf_lista_pagos_proveedores', compact('pagos_a_proveedores'));

                $file_name = time() . '_Pago_a_proveedor_temp.pdf';
                $pdf_pp_get = storage_path('app/public/pago_a_proveedor_pdf_listado_temporal/').$file_name;
                $pdf->save($pdf_pp_get);

                $pdfMerger = PDFMerger::init(); //Initialize the merger
                $pdfMerger->addPDF($pdf_pp_get, 'all');
                foreach ($pagos_a_proveedores as $pago_proveedor) {
                    if ($pago_proveedor->url_factura_pdf){
                        $url_factura_pdf = storage_path('app/public/').$pago_proveedor->url_factura_pdf;
                        $pdfMerger->addPDF($url_factura_pdf, 'all');
                    }
                    if ($pago_proveedor->url_resumen_pdf){
                        $url_resumen_pdf = storage_path('app/public/').$pago_proveedor->url_resumen_pdf;
                        $pdfMerger->addPDF($url_resumen_pdf, 'all');
                    }
                }
                $pdfMerger->merge();
                $pdfMerger->save($path_public.$file_name);

                $path = 'pago_a_proveedor_pdf_listado_temporal/';
                $archivo_url = $path . $file_name;
                @unlink($pdf_pp_get);

                $response = [
                    'status' => "OK",
                    'error' => false,
                    'message' => 'PDF generado con éxito.',
                    'archivo_url' => $archivo_url,
                ];
            }
            return response()->json($response);

        } catch (\Exception $e) {
            Log::info('Pago a proveedores | Exception => ' . $e->getMessage() . ' in ' . $e->getLine());
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'message' => 'Ocurrio un error al generar el pdf, contacte al administrador del sistema.',
            ]);
        }
    }
}
