<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\IvaImpuesto;
use Illuminate\Http\Request;
use DataTables;
use Illuminate\Support\Facades\Auth;

class IvaImpuestoController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();

        $catalogo_de_iva_impuesto_edit = $user->can('catalogo_de_iva_impuesto_edit');
        $catalogo_de_iva_impuesto_destroy = $user->can('catalogo_de_iva_impuesto_destroy');

        return view('admin.catalogos.iva.lista', compact('catalogo_de_iva_impuesto_edit', 'catalogo_de_iva_impuesto_destroy'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('admin.catalogos.iva.nuevo');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'iva' => 'required|numeric',
            'nombre' => 'required',
        ]);

        $iva_impuesto = new IvaImpuesto();
        $iva_impuesto->iva = $request->input('iva');
        $iva_impuesto->nombre = $request->input('nombre');
        $iva_impuesto->descripcion = $request->input('descripcion');
        $iva_impuesto->save();

        return redirect()->route('catalogo-de-iva-impuesto.index')->withSuccess('IVA guardado con éxito.');

    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $iva_impuesto = IvaImpuesto::findOrFail($id);
        return view('admin.catalogos.iva.editar', compact("iva_impuesto"));

    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validated = $request->validate([
            'iva' => 'required|numeric',
            'nombre' => 'required',
        ]);

        $iva_impuesto = IvaImpuesto::findOrFail($id);
        $iva_impuesto->iva = $request->input('iva');
        $iva_impuesto->nombre = $request->input('nombre');
        $iva_impuesto->descripcion = $request->input('descripcion');
        $iva_impuesto->update();

        return redirect()->route('catalogo-de-iva-impuesto.index')->withSuccess('IVA actualizado con éxito.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $iva_impuesto = IvaImpuesto::findOrFail($id);
            $iva_impuesto->delete();

            return response()->json(['error' => false, 'message' => 'Registro eliminado.']);
        } catch (\Exception $ex) {
            return response()->json([
                'error' => true,
                'message' => $ex->getMessage(),
            ]);
        }
    }

    public function get_Lista_Iva(Request $request)
    {
        if ($request->ajax()) {
            $iva_impuesto = IvaImpuesto::whereNull('deleted_at')
                ->orderBy('id', 'desc')->get();

            return DataTables::of($iva_impuesto)->make(true);
        }
    }

    /**
     * Se obtiene Lista de IVA para Select - en módulo Orden de Compras
     * El iva compuesto: id@iva - se usa para calculos en precios productos
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function get_Lista_Iva_Impuesto_Select(Request $request)
    {
        $iva_impuesto = [];
        if($request->has('search')){
            $search = $request->search;
            $iva_impuesto = IvaImpuesto::select('id', 'iva', 'nombre', 'descripcion')
                ->whereNull('deleted_at')
                ->where('iva', 'LIKE', $search.'%')
                ->orderBy('iva', 'asc')
                ->get();
        }else{
            $iva_impuesto = IvaImpuesto::select('id', 'iva', 'nombre', 'descripcion')
                ->whereNull('deleted_at')
                ->orderBy('iva', 'asc')
                ->get();
        }

        if(!$iva_impuesto->isEmpty()){
            $valid_tags = [];
            foreach ($iva_impuesto as $item) {
                //IVA compuesto
                $valid_tags[] = ['id' => $item->id .'@'. $item->iva, 'text' => $item->nombre];
            }
            return response()->json($valid_tags);
        }else{
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'message' => 'No se encontraron resultados'
            ]);
        }
    }

    public function get_Lista_Iva_Select_Simple(Request $request)
    {
        $iva_impuesto = [];
        if($request->has('search')){
            $search = $request->search;
            $iva_impuesto = IvaImpuesto::select('id', 'iva', 'nombre', 'descripcion')
                ->whereNull('deleted_at')
                ->where('iva', 'LIKE', $search.'%')
                ->orderBy('iva', 'asc')
                ->get();
        }else{
            $iva_impuesto = IvaImpuesto::select('id', 'iva', 'nombre', 'descripcion')
                ->whereNull('deleted_at')
                ->orderBy('iva', 'asc')
                ->get();
        }

        if(!$iva_impuesto->isEmpty()){
            $valid_tags = [];
            foreach ($iva_impuesto as $item) {
                $valid_tags[] = ['id' => $item->id, 'text' => $item->nombre];
            }
            return response()->json($valid_tags);
        }else{
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'message' => 'No se encontraron resultados'
            ]);
        }
    }

}
