<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Unidad;
use Illuminate\Http\Request;
use DataTables;
use Illuminate\Support\Facades\Auth;

class UnidadesController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();

        $catalogo_de_unidades_edit = $user->can('catalogo_de_unidades_edit');
        $catalogo_de_unidades_destroy = $user->can('catalogo_de_unidades_destroy');

        return view('admin.catalogos.unidades.lista', compact('catalogo_de_unidades_edit', 'catalogo_de_unidades_destroy'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('admin.catalogos.unidades.nuevo');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'nombre' => 'required|min:3',
        ]);

        $unidad = new Unidad();
        $unidad->nombre = $request->input('nombre');
        $unidad->descripcion = $request->input('descripcion');
        $unidad->save();

        return redirect()->route('catalogo-de-unidades.index')->withSuccess('Unidad guardada con éxito.');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $unidad = Unidad::findOrFail($id);
        return view('admin.catalogos.unidades.editar', compact("unidad"));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validated = $request->validate([
            'nombre' => 'required|min:3',
        ]);

        $unidad = Unidad::findOrFail($id);
        $unidad->nombre = $request->input('nombre');
        $unidad->descripcion = $request->input('descripcion');
        $unidad->update();

        return redirect()->route('catalogo-de-unidades.index')->withSuccess('Unidad actualizada con éxito.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $unidad = Unidad::findOrFail($id);
            $unidad->delete();

            return response()->json(['error' => false, 'message' => 'Registro eliminado.']);
        } catch (\Exception $ex) {
            return response()->json([
                'error' => true,
                'message' => $ex->getMessage(),
            ]);
        }
    }

    public function get_Lista_Unidades(Request $request)
    {
        if ($request->ajax()) {
            $unidades = Unidad::whereNull('deleted_at')
                ->orderBy('id', 'desc')->get();

            return DataTables::of($unidades)->make(true);
        }
    }

    public function get_Lista_Unidades_Select(Request $request)
    {
        $unidades = [];
        if($request->has('search')){
            $search = $request->search;

            $unidades = Unidad::select('id', 'nombre')
                ->whereNull('deleted_at')
                ->where('nombre', 'LIKE', '%'.$search.'%')
                ->orderBy('nombre', 'asc')
                ->get();
        }else{
            $unidades = Unidad::select('id', 'nombre')
                ->whereNull('deleted_at')
                ->orderBy('nombre', 'asc')
                ->get();
        }

        if(!$unidades->isEmpty()){
            $valid_tags = [];
            foreach ($unidades as $item) {
                $valid_tags[] = ['id' => $item->id, 'text' => $item->nombre];
            }
            return response()->json($valid_tags);
        }else{
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'message' => 'No se encontraron resultados'
            ]);
        }
    }
}
