<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Proveedor;
use App\Models\ProveedorCuentaBancaria;
use App\Models\User;
use Illuminate\Http\Request;
use DataTables;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class ProveedoresController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();

        $catalogo_de_proveedores_edit = $user->can('catalogo_de_proveedores_edit');
        $catalogo_de_proveedores_destroy = $user->can('catalogo_de_proveedores_destroy');

        return view('admin.catalogos.proveedores.lista', compact('catalogo_de_proveedores_edit', 'catalogo_de_proveedores_destroy'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('admin.catalogos.proveedores.nuevo');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $response = [];
        $dataAll = $request->all();
        $cError = ""; //Errores
        $resultadoErrors = []; //Lista de Errores

        $validator = \Validator::make($dataAll, [
            'razon_social' => 'required',
            'nombre_comercial' => 'required',
            'rfc' => 'required',
            'iva_id' => 'required',
            'cuentas_bancarias_data' => 'required|array|min:1',
        ]);

        if ($validator->fails()) {
            $response = [
                'status' => "ERROR",
                'error' => true,
                'message' => 'Campos obligatorios',
                'errors' => $validator->errors()->all()
            ];
        } else {
            $proveedor = new Proveedor();
            $proveedor->razon_social = $dataAll['razon_social'];
            $proveedor->nombre_comercial = $dataAll['nombre_comercial'];
            $proveedor->rfc = $dataAll['rfc'];
            $proveedor->direccion = $dataAll['direccion'];
            $proveedor->telefono = $dataAll['telefono'];
            $proveedor->email = $dataAll['email'];
            $proveedor->comentarios = $dataAll['comentarios'];
            $proveedor->iva_impuestos_id = $dataAll['iva_id'];
            $proveedor->aplica_isr = $dataAll['aplica_isr'];
            if ($dataAll['aplica_isr'] == 1){
                $proveedor->aplica_monto_isr = 0.0125;
            }
            $proveedor->contacto = $dataAll['contacto'];

            if ($proveedor->save()){

                //lista cuentas bancarias
                $cuentas_bancarias_data = $dataAll['cuentas_bancarias_data'];
                foreach ($cuentas_bancarias_data as $cuenta_bancaria) {
                    $proveedor_cuenta_bancaria = new ProveedorCuentaBancaria();
                    $proveedor_cuenta_bancaria->proveedor_id = $proveedor->id;
                    $proveedor_cuenta_bancaria->cuenta_bancaria = $cuenta_bancaria['1'];
                    $proveedor_cuenta_bancaria->clabe_interbancaria = $cuenta_bancaria['2'];
                    $proveedor_cuenta_bancaria->banco = $cuenta_bancaria['3'];
                    if (!$proveedor_cuenta_bancaria->save()){
                        array_push($resultadoErrors, 'Ocurrio un error al intentar guardar la cuenta bancaria: '.$cuenta_bancaria['1']);
                    }
                }

                if (!empty($resultadoErrors)) {
                    $response = [
                        'status' => "ERROR",
                        'error' => true,
                        'message' => 'Ocurrió un error al guardar el proveedor.',
                        'errors' => $resultadoErrors
                    ];
                } else {
                    $response = [
                        'status' => "OK",
                        'error' => false,
                        'message' => 'Proveedor guardado correctamente.',
                        'data' => $proveedor
                    ];
                }

            } else {
                $response = [
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'Ocurrió un error al guardar el proveedor',
                    'errors' => 'New Proveedor: error al guardar el proveedor.'
                ];
            }

        }

        return response()->json($response);
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $proveedor = Proveedor::with(['ProveedorCuentaBancaria', 'ivaImpuesto'])->findOrFail($id);

        $option_default_iva = "";
        if (!empty($proveedor->ivaImpuesto)){
            $option_default_iva = "<option value='" . $proveedor->ivaImpuesto->id . "' selected='selected'>" . $proveedor->ivaImpuesto->nombre . "</option>";
        }

        return view('admin.catalogos.proveedores.editar', compact(["proveedor", "option_default_iva"]));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $response = [];
        $dataAll = $request->all();
        $cError = ""; //Errores
        $resultadoErrors = []; //Lista de Errores

        $validator = \Validator::make($dataAll, [
            'razon_social' => 'required',
            'nombre_comercial' => 'required',
            'rfc' => 'required',
            'iva_id' => 'required',
            'cuentas_bancarias_data' => 'required|array|min:1',
        ]);

        if ($validator->fails()) {
            $response = [
                'status' => "ERROR",
                'error' => true,
                'message' => 'Campos obligatorios',
                'errors' => $validator->errors()->all()
            ];
        } else {
            $proveedor = Proveedor::findOrFail($id);
            $proveedor->razon_social = $dataAll['razon_social'];
            $proveedor->nombre_comercial = $dataAll['nombre_comercial'];
            $proveedor->rfc = $dataAll['rfc'];
            $proveedor->direccion = $dataAll['direccion'];
            $proveedor->telefono = $dataAll['telefono'];
            $proveedor->email = $dataAll['email'];
            $proveedor->comentarios = $dataAll['comentarios'];
            $proveedor->iva_impuestos_id = $dataAll['iva_id'];
            $proveedor->aplica_isr = $dataAll['aplica_isr'];
            if ($dataAll['aplica_isr'] == 1){
                $proveedor->aplica_monto_isr = 0.0125;
            }
            $proveedor->contacto = $dataAll['contacto'];

            if ($proveedor->update()) {
                //Dar de baja las cuentas bancarias actuales
                $cuentas_bancarias = ProveedorCuentaBancaria::where('proveedor_id', $id)->whereNull('deleted_at')->get();
                if (!empty($cuentas_bancarias)) {
                    foreach ($cuentas_bancarias as $cuenta_bancaria) {
                        if (!$cuenta_bancaria->delete()){
                            array_push($resultadoErrors, 'Ocurrio un error al intentar eliminar la cuenta bancaria con clave: '.$cuenta_bancaria['1']);
                        }
                    }
                }

                //lista cuentas bancarias
                $cuentas_bancarias_data = $dataAll['cuentas_bancarias_data'];
                foreach ($cuentas_bancarias_data as $cuenta_bancaria) {
                    //Dar de alta las cuentas bancarias
                    $proveedor_cuenta_bancaria = new ProveedorCuentaBancaria();
                    $proveedor_cuenta_bancaria->proveedor_id = $proveedor->id;
                    $proveedor_cuenta_bancaria->cuenta_bancaria = $cuenta_bancaria['2'];
                    $proveedor_cuenta_bancaria->clabe_interbancaria = $cuenta_bancaria['3'];
                    $proveedor_cuenta_bancaria->banco = $cuenta_bancaria['4'];
                    if (!$proveedor_cuenta_bancaria->save()){
                        array_push($resultadoErrors, 'Ocurrio un error al intentar guardar la cuenta bancaria: '.$cuenta_bancaria['2']);
                    }
                }

                if (!empty($resultadoErrors)) {
                    $response = [
                        'status' => "ERROR",
                        'error' => true,
                        'message' => 'Ocurrió un error al actualizar los datos del proveedor.',
                        'errors' => $resultadoErrors
                    ];
                } else {
                    $response = [
                        'status' => "OK",
                        'error' => false,
                        'message' => 'Los datos del proveedor fueron actualizados correctamente.',
                        'data' => $proveedor
                    ];
                }

            } else {
                $response = [
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'Ocurrió un error al actualizar los datos del proveedor',
                    'errors' => 'Update Proveedor: error al actualizar los datos del proveedor.'
                ];
            }
        }

        return response()->json($response);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $proveedor = Proveedor::findOrFail($id);
            $proveedor->delete();

            return response()->json(['error' => false, 'message' => 'Registro eliminado.']);
        } catch (\Exception $ex) {
            return response()->json([
                'error' => true,
                'message' => $ex->getMessage(),
            ]);
        }
    }

    public function get_Lista_Proveedores(Request $request)
    {
        if ($request->ajax()) {
            $proveedores = Proveedor::with('ivaImpuesto')
                ->whereNull('deleted_at')
                ->orderBy('id', 'desc')->get();

            return DataTables::of($proveedores)->make(true);
        }
    }

    public function get_Lista_Proveedores_Select(Request $request)
    {
        $proveedores = [];
        if($request->has('search')) {
            $search = $request->search;

            $proveedores = Proveedor::select('id', 'razon_social', 'nombre_comercial', 'rfc', 'direccion', 'telefono', 'email')
                ->whereNull('deleted_at')
                ->where('razon_social', 'LIKE', '%'.$search.'%')
                ->orderBy('razon_social', 'asc')
                ->get();
        }else{
            $proveedores = Proveedor::select('id', 'razon_social', 'nombre_comercial', 'rfc', 'direccion', 'telefono', 'email')
                ->whereNull('deleted_at')
                ->orderBy('razon_social', 'asc')
                ->get();
        }

        if(!$proveedores->isEmpty()){
            $valid_tags = [];
            foreach ($proveedores as $item) {
                $valid_tags[] = ['id' => $item->id, 'text' => $item->razon_social];
            }
            return response()->json($valid_tags);
        }else{
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'message' => 'No se encontraron resultados'
            ]);
        }
    }

    public function getProveedorById($id)
    {
        try {
            $proveedor = Proveedor::with(['ProveedorCuentaBancaria', 'ivaImpuesto'])
                ->find($id);

            return response()->json([
                'error' => false,
                'mensajes' => 'Proveedor encontrado',
                'proveedor' => $proveedor
            ]);

        } catch(\Exception $ex){
            return response()->json([
                'error' => true,
                'mensajes' => $ex->getMessage() . ' in line ' . $ex->getLine(),
            ]);
        }
    }

}
