<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Helpers\Consts;
use App\Mail\MessageReceived;
use App\Models\ComentarioMaterial;
use App\Models\Estatus;
use App\Models\IvaImpuesto;
use App\Models\OrdenCompra;
use App\Models\OrdenCompraAnticipo;
use App\Models\RelSolicitudProductoComentarioMaterial;
use App\Models\solicitud_clon\ComentarioMaterialClon;
use App\Models\solicitud_clon\RelSolicitudProductoComentarioMaterialClon;
use App\Models\solicitud_clon\SolicitudCompraClon;
use App\Models\solicitud_clon\SolicitudCompraProductoClon;
use App\Models\solicitud_material\RelSolicitudCompraProductoPrecio;
use App\Models\SolicitudCompra;
use App\Models\SolicitudCompraProducto;
use App\Models\TipoPago;
use Carbon\Carbon;
use Illuminate\Http\Request;
use DataTables;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use PDFSnappy;
use LynX39\LaraPdfMerger\Facades\PdfMerger;

class OrdenesComprasController extends Controller
{
    protected $path_archivo_pdf;
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('can:ordenes_de_compras_index')->only('index');
        $this->middleware('can:ordenes_de_compras_create')->only('create', 'store');
        $this->middleware('can:ordenes_de_compras_edit')->only('edit', 'update');
        $this->middleware('can:ordenes_de_compras_destroy')->only('destroy');
        $this->middleware('can:ordenes_de_compras_show')->only('show');
        $this->path_archivo_pdf = ('orden_compra_pdf');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();

        $ordenes_de_compras_show = $user->can('ordenes_de_compras_show');
        $ordenes_de_compras_edit = $user->can('ordenes_de_compras_edit');
        $ordenes_de_compras_destroy  = $user->can('ordenes_de_compras_destroy');
        $ordenes_de_compras_export  = $user->can('ordenes_de_compras_export');
        $ordenes_de_compras_send_email = $user->can('ordenes_de_compras_send_email');
        $ordenes_de_compras_authorized = $user->can('ordenes_de_compras_authorized');
        $ordenes_de_compras_enviar_a_pago = $user->can('ordenes_de_compras_enviar_a_pago');

        //Estatus Por autorizar
        $estatus_por_autorizar = Estatus::where('clave', 'por_autorizar')->whereNull('deleted_at')->first();
        //Estatus: Autorizado
        $estatus_autorizado = Estatus::where('clave', 'autorizado')->whereNull('deleted_at')->first();

        return view('admin.ordenes_de_compras.lista', compact(
            'estatus_por_autorizar',
            'estatus_autorizado',
            'ordenes_de_compras_show',
            'ordenes_de_compras_edit',
            'ordenes_de_compras_destroy',
            'ordenes_de_compras_export',
            'ordenes_de_compras_send_email',
            'ordenes_de_compras_authorized',
            'ordenes_de_compras_enviar_a_pago',
        ));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $usarSolicitud = filter_var(request('usarSolicitud', true), FILTER_VALIDATE_BOOLEAN);
        return view('admin.ordenes_de_compras.nuevo', ['usarSolicitud' => $usarSolicitud]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();
            $response = [];
            $dataAll = $request->all();
            $resultadoErrors = []; //Lista de Errores

            $validator = \Validator::make($dataAll, [
                'proveedor_id' => 'required|integer',
                'solicitud_compra_id' => 'required|integer',
                'productos_data' => 'required|array|min:1',
                'tipo_pago_id' => 'required|integer',
            ]);

            if ($validator->fails()) {
                array_push($resultadoErrors, 'Campos obligatorios: ' . implode(',', $validator->errors()->all()));
            } else {
                $solicitud_compra_id = $dataAll['solicitud_compra_id'];

                $orden_compra = new OrdenCompra();
                $orden_compra->solicitud_compra_id = $solicitud_compra_id;
                $orden_compra->proveedor_id = $dataAll['proveedor_id'];
                $orden_compra->vendedor = $dataAll['vendedor'];
                $orden_compra->metodo_envio = $dataAll['metodo_envio'];
                $orden_compra->conticiones_de_pago = $dataAll['conticiones_de_pago'];
                $orden_compra->comentarios = $dataAll['comentarios'];
                $orden_compra->fecha_orden = Carbon::now()->format('Y-m-d');
                $orden_compra->clave_version = Consts::ORDEN_COMPRA_V2;

                if ($dataAll['fecha_entrega']) {
                    $orden_compra->fecha_entrega = (Carbon::createFromFormat('d/m/Y', $dataAll['fecha_entrega']))->format('Y-m-d');
                }
                if ($dataAll['fecha_embarque']) {
                    $orden_compra->fecha_embarque = (Carbon::createFromFormat('d/m/Y', $dataAll['fecha_embarque']))->format('Y-m-d');
                }

                //Crear O.C. con Estatus = Por autorizar
                $estatus_por_autorizar = Estatus::where('clave', 'por_autorizar')->whereNull('deleted_at')->first();
                $orden_compra->estatus_id = $estatus_por_autorizar->id;

                //Materiales: Importe sin impuestos
                $orden_compra->importe_sin_impuestos = $dataAll['importe_sin_impuestos'];
                if ($dataAll['iva']) {
                    $iva_impuesto = IvaImpuesto::where('iva', $dataAll['iva'])->first();
                    $orden_compra->iva_impuestos_id = $iva_impuesto->id;
                }
                //Materiales IVA
                $orden_compra->monto_iva = $dataAll['monto_de_iva'];
                //Materiales ISR
                $orden_compra->monto_isr = $dataAll['monto_isr'];
                //Materiales Total
                $orden_compra->total = $dataAll['total'];
                $orden_compra->total_letras = $dataAll['total_letras'];

                $orden_compra->emails = $dataAll['emails'];
                $orden_compra->email_asunto = $dataAll['email_asunto'];
                $orden_compra->email_contenido = $dataAll['email_contenido'];

                $orden_compra->tipo_pago_id = $dataAll['tipo_pago_id'];

                if ($orden_compra->save()) {
                    $foliador = $this->getFoliador($orden_compra->id);
                    $orden_compra->codigo = $foliador;
                    $orden_compra->codigo_nombre = 'orden-de-compra-' . $foliador;

                    if ($orden_compra->update()) {
                        //lista materiales y precios
                        $productos_data = $dataAll['productos_data'];

                        foreach ($productos_data as $producto) {
                            $existingProducto = SolicitudCompraProducto::where('solicitud_compra_id', $orden_compra->solicitud_compra_id)
                                ->where('producto_id', $producto['2'])
                                ->where('num_duplicado_order', $producto['16'])
                                ->whereNull('deleted_at')
                                ->first();

                            if ($existingProducto) {
                                //Actualiza Material existente
                                $existingProducto->producto_id = $producto['2'];
                                $existingProducto->unidad_id = $producto['4'];
                                $existingProducto->cantidad = $producto['6'];
                                $existingProducto->area_de_uso = $producto['7'];
                                $existingProducto->marca = $producto['8'];
                                $existingProducto->modelo = $producto['9'];
                                $existingProducto->observaciones = $producto['10'];
                                $existingProducto->num_duplicado_order = $producto['16'];
                                //Precios se usa en la V.1.0
//                                $existingProducto->precio_lista = $producto['11'];
//                                $existingProducto->precio_unitario = $producto['12'];
//                                $existingProducto->importe = $producto['13'];
//                                $existingProducto->descuento = $producto['14'];
//                                $existingProducto->subtotal = $producto['15'];
                                //Fin-Precios se usa en la V.1.0

                                if ($existingProducto->save()) {
                                    //Validar si Existe el Precio del Material para la O.C.
                                    $existe_Precio = RelSolicitudCompraProductoPrecio::where('solicitud_compra_producto_id', $existingProducto->id)
                                        ->where('orden_compra_id', $orden_compra->id)->whereNull('deleted_at')->first();

                                    if (empty($existe_Precio)) {
                                        //Guardar Precio de material
                                        $precio_material = new RelSolicitudCompraProductoPrecio();
                                        $precio_material->solicitud_compra_producto_id = $existingProducto->id;
                                        $precio_material->orden_compra_id = $orden_compra->id;
                                        $precio_material->precio_lista = $producto['11'];
                                        $precio_material->precio_unitario = $producto['12'];
                                        $precio_material->importe = $producto['13'];
                                        $precio_material->descuento = $producto['14'];
                                        $precio_material->subtotal = $producto['15'];
                                        $precio_material->created_by = auth()->id();
                                        $precio_material->updated_by = auth()->id();
                                        if (!$precio_material->save()) {
                                            array_push($resultadoErrors, 'Error al guardar el Precio del Material con clave: '.$producto['2']);
                                        }
                                    }
                                } else {
                                    array_push($resultadoErrors, 'Error al guardar el Material con clave: '.$producto['2']);
                                }

                            } else {
                                //NO Existe el Material y crea material
                                $newProducto = new SolicitudCompraProducto();
                                $newProducto->solicitud_compra_id = $orden_compra->solicitud_compra_id;
                                $newProducto->producto_id = $producto['2'];
                                $newProducto->unidad_id = $producto['4'];
                                $newProducto->cantidad = $producto['6'];
                                $newProducto->area_de_uso = $producto['7'];
                                $newProducto->marca = $producto['8'];
                                $newProducto->modelo = $producto['9'];
                                $newProducto->observaciones = $producto['10'];
                                $newProducto->num_duplicado_order = $producto['16'];
                                //Precios se usa en la V.1.0
//                                $newProducto->precio_lista = $producto['11'];
//                                $newProducto->precio_unitario = $producto['12'];
//                                $newProducto->importe = $producto['13'];
//                                $newProducto->descuento = $producto['14'];
//                                $newProducto->subtotal = $producto['15'];
                                //Fin-Precios se usa en la V.1.0

                                if ($newProducto->save()) {
                                    //Guardar Precio de material
                                    $precio_material = new RelSolicitudCompraProductoPrecio();
                                    $precio_material->solicitud_compra_producto_id = $newProducto->id;
                                    $precio_material->orden_compra_id = $orden_compra->id;
                                    $precio_material->precio_lista = $producto['11'];
                                    $precio_material->precio_unitario = $producto['12'];
                                    $precio_material->importe = $producto['13'];
                                    $precio_material->descuento = $producto['14'];
                                    $precio_material->subtotal = $producto['15'];
                                    $precio_material->created_by = auth()->id();
                                    $precio_material->updated_by = auth()->id();
                                    if (!$precio_material->save()) {
                                        array_push($resultadoErrors, 'Error al guardar el Precio del Material con clave: '.$producto['2']);
                                    }
                                } else {
                                    array_push($resultadoErrors, 'Error al guardar nuevo Material con clave: '.$producto['2']);
                                }
                            }

                            // Guardar el ID del producto procesado
                            $productos_ids[] = $producto['2'];
                        }

                        // 2. Eliminar registros que no están en la lista actualizada
                        SolicitudCompraProducto::where('solicitud_compra_id', $orden_compra->solicitud_compra_id)
                            ->whereNotIn('producto_id', $productos_ids) // Solo los que no estén en la lista
                            ->delete();

                        /**
                         * @deprecated YA NO SE USA PORQUE EL ESTATUS DE LA S.C. SE MANTIENE EN AUTORIZADO V.2.0
                         * Actualizar Estatus de Solicitud de Compra en: Por autorizar
                         * @author russel.cruz
                         */
//                        $solicitud_compra = SolicitudCompra::findOrFail($solicitud_compra_id);
//                        $solicitud_compra->estatus_id = 2; //Por autorizar
//                        $solicitud_compra->update();

                        //Lista-Anticipos
                        $anticipos_data = $dataAll['anticipos_data'];
                        if (!empty($anticipos_data)) {
                            foreach ($anticipos_data as $anticipo) {
                                $nuevo_anticipo = new OrdenCompraAnticipo();
                                $nuevo_anticipo->orden_compra_id = $orden_compra->id;
                                $nuevo_anticipo->comentario = $anticipo['1'];
                                $nuevo_anticipo->anticipo = $anticipo['2'];
                                $nuevo_anticipo->anticipo_letras = $anticipo['3'];
                                $nuevo_anticipo->forma_de_pago = $anticipo['4'];
                                if (!empty($anticipo['5'])) {
                                    $nuevo_anticipo->fecha_anticipo = (Carbon::createFromFormat('d/m/Y', $anticipo['5']))->format('Y-m-d');
                                } else {
                                    $nuevo_anticipo->fecha_anticipo = Carbon::now()->format('Y-m-d');
                                }
                                $nuevo_anticipo->save();
                            }
                        }
                    } else {
                        array_push($resultadoErrors, 'No se pudó actualizar la Orden de compra.');
                    }
                } else {
                    array_push($resultadoErrors, 'No se pudó guardar la Orden de compra.');
                }
            }

            if (empty($resultadoErrors)) {
                DB::commit();
                $response = [
                    'status' => "OK",
                    'error' => false,
                    'message' => 'Orden de compra guardada correctamente.',
                    'errors' => ''
                ];
            } else {
                DB::rollBack();
                $response = [
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'Ocurrio un error al guardar la Orden de compra.',
                    'errors' => $resultadoErrors
                ];
            }

            return response()->json($response);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::info('Exception => ' . $e->getMessage() . ' in ' . $e->getLine());
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'message' => 'Ocurrio un error al guardar la Orden de compra, contacte al administrador del sistema.',
                'errors' => $e->getMessage() . ' in ' . $e->getLine()
            ]);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //Estatus Por autorizar
        $estatus_por_autorizar = Estatus::where('clave', 'por_autorizar')->whereNull('deleted_at')->first();
        //Estatus: Autorizado
        $estatus_autorizado = Estatus::where('clave', 'autorizado')->whereNull('deleted_at')->first();

        $orden_compra_materiales = OrdenCompra::with([
            'solicitudCompra.solicitudCompraProducto',
        ])->where('id', $id)
            ->whereNull('deleted_at')
            ->first();

        //Al dar de alta el Material desde la solicitud existente no se creo su Precio
        //Crear el Precio del Material SOLO si la O.C. es version 2 y estatus Por Autorizar y no tiene clon
        if ($orden_compra_materiales->clave_version === Consts::ORDEN_COMPRA_V2 && $orden_compra_materiales->estatus_id === $estatus_por_autorizar->id && $orden_compra_materiales->solicitud_compra_clon_id == null) {
            foreach ($orden_compra_materiales->solicitudCompra->solicitudCompraProducto as $material) {
                $precio_del_material_existe = RelSolicitudCompraProductoPrecio::where('solicitud_compra_producto_id', $material->id)
                    ->where('orden_compra_id', $orden_compra_materiales->id)
                    ->whereNull('deleted_at')->first();

                if (empty($precio_del_material_existe)) {
                    //Crear precio del material
                    $precio_material = new RelSolicitudCompraProductoPrecio();
                    $precio_material->solicitud_compra_producto_id = $material->id;
                    $precio_material->orden_compra_id = $orden_compra_materiales->id;
                    $precio_material->precio_lista = 0;
                    $precio_material->precio_unitario = 0;
                    $precio_material->importe = 0;
                    $precio_material->descuento = 0;
                    $precio_material->subtotal = 0;
                    $precio_material->created_by = auth()->id();
                    $precio_material->updated_by = auth()->id();
                    $precio_material->save();
                }
            }
        }

        //Obtener la orden de compras
        $orden_compra = OrdenCompra::with([
            'proveedor',
            'ivaImpuesto',
            'estatus',
            'ordenCompraAnticipo',
            'solicitudCompra',
            'solicitudCompra.obra',
            'solicitudCompra.ingeniero',
            'solicitudCompra.solicitudCompraProducto',
            'solicitudCompra.solicitudCompraProducto.producto',
            'solicitudCompra.solicitudCompraProducto.unidad',
            'SolicitudCompraClon',
            'SolicitudCompraClon.obra',
            'SolicitudCompraClon.ingeniero',
            'SolicitudCompraClon.SolicitudCompraProductoClon',
            'SolicitudCompraClon.SolicitudCompraProductoClon.producto',
            'SolicitudCompraClon.SolicitudCompraProductoClon.unidad',
        ])->where('id', $id)
            ->whereNull('deleted_at')
            ->first();

        if ($orden_compra->clave_version === Consts::ORDEN_COMPRA_V2 && $orden_compra->estatus_id === $estatus_por_autorizar->id && $orden_compra->solicitud_compra_clon_id == null) {
            foreach ($orden_compra->solicitudCompra->solicitudCompraProducto as $material) {
                //Obtiene el precio del material
                $precio_del_material = RelSolicitudCompraProductoPrecio::where('solicitud_compra_producto_id', $material->id)
                    ->where('orden_compra_id', $orden_compra->id)
                    ->whereNull('deleted_at')->first();

                //Setea el precio al material
                $material->rel_solicitud_compra_producto_precio = $precio_del_material;
            }
        }

        $tipo_pago = TipoPago::select('id', 'nombre')
            ->whereNull('deleted_at')
            ->orderBy('id', 'asc')
            ->get();

        $existe_tipo_pago_selected = false;
        $options_tipoPago = "<option>Selecciona una opción</option>";
        foreach ($tipo_pago as $tipo) {
            if ($tipo->id == $orden_compra->tipo_pago_id) {
                $existe_tipo_pago_selected = true;
                $options_tipoPago .= "<option value='" . $tipo->id . "' selected>" . $tipo->nombre . "</option>";
            } else {
                $options_tipoPago .= "<option value='" . $tipo->id . "'>" . $tipo->nombre . "</option>";
            }
        }

        $orden_compra->existe_tipo_pago_selected = $existe_tipo_pago_selected;
        $orden_compra->tiene_pagos_anticipos = (count($orden_compra->ordenCompraAnticipo)) ? true : false;

        return view('admin.ordenes_de_compras.info', compact('orden_compra', 'options_tipoPago', 'estatus_por_autorizar', 'estatus_autorizado'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //Estatus Por autorizar
        $estatus_por_autorizar = Estatus::where('clave', 'por_autorizar')->whereNull('deleted_at')->first();
        //Estatus: Autorizado
        $estatus_autorizado = Estatus::where('clave', 'autorizado')->whereNull('deleted_at')->first();
        $user = Auth::user();
        //Role Gloria
        $role_ing_jefe_gloria = $user->hasRole('ingeniero-jefe'); // ? 1 : 0;

        $orden_compra_materiales = OrdenCompra::with([
            'solicitudCompra.solicitudCompraProducto',
        ])->where('id', $id)
            ->whereNull('deleted_at')
            ->first();

        //Al dar de alta el Material desde la solicitud existente no se creo su Precio
        //Crear el Precio del Material SOLO si la O.C. es version 2 y estatus Por Autorizar y no tiene clon
        if ($orden_compra_materiales->clave_version === Consts::ORDEN_COMPRA_V2 && $orden_compra_materiales->estatus_id === $estatus_por_autorizar->id && $orden_compra_materiales->solicitud_compra_clon_id == null) {
            foreach ($orden_compra_materiales->solicitudCompra->solicitudCompraProducto as $material) {
                $precio_del_material_existe = RelSolicitudCompraProductoPrecio::where('solicitud_compra_producto_id', $material->id)
                    ->where('orden_compra_id', $orden_compra_materiales->id)
                    ->whereNull('deleted_at')->first();

                if (empty($precio_del_material_existe)) {
                    //Crear precio del material
                    $precio_material = new RelSolicitudCompraProductoPrecio();
                    $precio_material->solicitud_compra_producto_id = $material->id;
                    $precio_material->orden_compra_id = $orden_compra_materiales->id;
                    $precio_material->precio_lista = 0;
                    $precio_material->precio_unitario = 0;
                    $precio_material->importe = 0;
                    $precio_material->descuento = 0;
                    $precio_material->subtotal = 0;
                    $precio_material->created_by = auth()->id();
                    $precio_material->updated_by = auth()->id();
                    $precio_material->save();
                }
            }
        }

        //Obtener la orden de compras
        $orden_compra = OrdenCompra::with([
            'proveedor',
            'ivaImpuesto',
            'estatus',
            'ordenCompraAnticipo',
            'solicitudCompra',
            'solicitudCompra.obra',
            'solicitudCompra.ingeniero',
            'solicitudCompra.solicitudCompraProducto',
            'solicitudCompra.solicitudCompraProducto.producto',
            'solicitudCompra.solicitudCompraProducto.unidad',
            'SolicitudCompraClon',
            'SolicitudCompraClon.obra',
            'SolicitudCompraClon.ingeniero',
            'SolicitudCompraClon.SolicitudCompraProductoClon',
            'SolicitudCompraClon.SolicitudCompraProductoClon.producto',
            'SolicitudCompraClon.SolicitudCompraProductoClon.unidad',
        ])->where('id', $id)
            ->whereNull('deleted_at')
            ->first();

        if ($orden_compra->clave_version === Consts::ORDEN_COMPRA_V2 && $orden_compra->estatus_id === $estatus_por_autorizar->id && $orden_compra->solicitud_compra_clon_id == null) {
            foreach ($orden_compra->solicitudCompra->solicitudCompraProducto as $material) {
                //Obtiene el precio del material
                $precio_del_material = RelSolicitudCompraProductoPrecio::where('solicitud_compra_producto_id', $material->id)
                    ->where('orden_compra_id', $orden_compra->id)
                    ->whereNull('deleted_at')->first();

                //Setea el precio al material
                $material->rel_solicitud_compra_producto_precio = $precio_del_material;
            }
        }

        $tipo_pago = TipoPago::select('id', 'nombre')
            ->whereNull('deleted_at')
            ->orderBy('id', 'asc')
            ->get();

        $existe_tipo_pago_selected = false;
        $options_tipoPago = "<option>Selecciona una opción</option>";
        foreach ($tipo_pago as $tipo) {
            if ($tipo->id == $orden_compra->tipo_pago_id) {
                $existe_tipo_pago_selected = true;
                $options_tipoPago .= "<option value='" . $tipo->id . "' selected>" . $tipo->nombre . "</option>";
            } else {
                $options_tipoPago .= "<option value='" . $tipo->id . "'>" . $tipo->nombre . "</option>";
            }
        }

        $orden_compra->existe_tipo_pago_selected = $existe_tipo_pago_selected;
        $orden_compra->tiene_pagos_anticipos = (count($orden_compra->ordenCompraAnticipo)) ? true : false;

        return view('admin.ordenes_de_compras.editar', compact('orden_compra', 'options_tipoPago', 'estatus_por_autorizar', 'estatus_autorizado', 'role_ing_jefe_gloria'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        try {
            DB::beginTransaction();
            $response = [];
            $dataAll = $request->all();
            $resultadoErrors = []; //Lista de Errores

            $validator = \Validator::make($dataAll, [
                //'proveedor_id' => 'required|integer',
                //'solicitud_compra_id' => 'required|integer',
                'productos_data' => 'required|array|min:1',
                'tipo_pago_id' => 'required|integer',
                'tipo_pago_id' => 'required|integer',
            ]);

            if ($validator->fails()) {
                array_push($resultadoErrors, 'Campos obligatorios: ' . implode(',', $validator->errors()->all()));
            } else {
                //Estatus Por autorizar
                $estatus_por_autorizar = Estatus::where('clave', 'por_autorizar')->whereNull('deleted_at')->first();
                //Estatus: Autorizado
                $estatus_autorizado = Estatus::where('clave', 'autorizado')->whereNull('deleted_at')->first();

                $orden_compra = OrdenCompra::findOrFail($id);
                $orden_compra->vendedor = $dataAll['vendedor'];
                $orden_compra->metodo_envio = $dataAll['metodo_envio'];
                $orden_compra->conticiones_de_pago = $dataAll['conticiones_de_pago'];
                $orden_compra->comentarios = $dataAll['comentarios'];
                $orden_compra->comentario_gloria = $dataAll['comentario_gloria'];

                if ($dataAll['fecha_entrega']) {
                    $orden_compra->fecha_entrega = (Carbon::createFromFormat('d/m/Y', $dataAll['fecha_entrega']))->format('Y-m-d');
                }
                if ($dataAll['fecha_embarque']) {
                    $orden_compra->fecha_embarque = (Carbon::createFromFormat('d/m/Y', $dataAll['fecha_embarque']))->format('Y-m-d');
                }

                //Materiales: Importe sin impuestos
                $orden_compra->importe_sin_impuestos = $dataAll['importe_sin_impuestos'];
                if ($dataAll['iva']) {
                    $iva_impuesto = IvaImpuesto::where('iva', $dataAll['iva'])->first();
                    $orden_compra->iva_impuestos_id = $iva_impuesto->id;
                }
                //Materiales IVA
                $orden_compra->monto_iva = $dataAll['monto_de_iva'];
                //Materiales ISR
                $orden_compra->monto_isr = $dataAll['monto_isr'];
                //Materiales Total
                $orden_compra->total = $dataAll['total'];
                $orden_compra->total_letras = $dataAll['total_letras'];

                $orden_compra->emails = $dataAll['emails'];
                $orden_compra->email_asunto = $dataAll['email_asunto'];
                $orden_compra->email_contenido = $dataAll['email_contenido'];

                $orden_compra->tipo_pago_id = $dataAll['tipo_pago_id'];

                if ($orden_compra->update()) {
                    //lista materiales y precios
                    $productos_data = $dataAll['productos_data'];

                    if ($orden_compra->clave_version === Consts::ORDEN_COMPRA_V1 || ($orden_compra->clave_version === Consts::ORDEN_COMPRA_V2 && $orden_compra->estatus_id === $estatus_por_autorizar->id && empty($orden_compra->solicitud_compra_clon_id))) {
                        //Actualizar O.C. en V1 o V2 y sin clon
                        foreach ($productos_data as $producto) {
                            $existingProducto = SolicitudCompraProducto::with('RelSolicitudCompraProductoPrecio')
                                ->where('solicitud_compra_id', $orden_compra->solicitud_compra_id)
                                ->where('producto_id', $producto['2'])
                                ->where('num_duplicado_order', $producto['16'])
                                ->whereNull('deleted_at')
                                ->first();

                            if ($existingProducto) {
                                //Actualiza Material existente
                                $existingProducto->unidad_id = $producto['4'];
                                $existingProducto->cantidad = $producto['6'];
                                $existingProducto->area_de_uso = $producto['7'];
                                $existingProducto->marca = $producto['8'];
                                $existingProducto->modelo = $producto['9'];
                                $existingProducto->observaciones = $producto['10'];
                                $existingProducto->num_duplicado_order = $producto['16'];

                                if ($orden_compra->clave_version === Consts::ORDEN_COMPRA_V1) {
                                    //Actualiza Precio para v1
                                    $existingProducto->precio_lista = $producto['11'];
                                    $existingProducto->precio_unitario = $producto['12'];
                                    $existingProducto->importe = $producto['13'];
                                    $existingProducto->descuento = $producto['14'];
                                    $existingProducto->subtotal = $producto['15'];
                                }

                                if ($existingProducto->update()) {
                                    if ($orden_compra->clave_version === Consts::ORDEN_COMPRA_V2) {
                                        //Validar si Existe el Precio del Material para la O.C.
                                        $existe_Precio = RelSolicitudCompraProductoPrecio::where('solicitud_compra_producto_id', $existingProducto->id)
                                            ->where('orden_compra_id', $orden_compra->id)->whereNull('deleted_at')->first();

                                        if (!empty($existe_Precio)){
                                            //Actualiza el Precio del Material
                                            $existe_Precio->precio_lista = $producto['11'];
                                            $existe_Precio->precio_unitario = $producto['12'];
                                            $existe_Precio->importe = $producto['13'];
                                            $existe_Precio->descuento = $producto['14'];
                                            $existe_Precio->subtotal = $producto['15'];
                                            $existe_Precio->updated_by = auth()->id();
                                            if (!$existe_Precio->update()) {
                                                array_push($resultadoErrors, 'Error al actualizar el Precio del Material con clave: '.$producto['2']);
                                            }
                                        } else {
                                            //No existe el Precio Crear Precio de material
                                            $precio_material_new = new RelSolicitudCompraProductoPrecio();
                                            $precio_material_new->solicitud_compra_producto_id = $existingProducto->id;
                                            $precio_material_new->orden_compra_id = $orden_compra->id;
                                            $precio_material_new->precio_lista = $producto['11'];
                                            $precio_material_new->precio_unitario = $producto['12'];
                                            $precio_material_new->importe = $producto['13'];
                                            $precio_material_new->descuento = $producto['14'];
                                            $precio_material_new->subtotal = $producto['15'];
                                            $precio_material_new->created_by = auth()->id();
                                            $precio_material_new->updated_by = auth()->id();
                                            if (!$precio_material_new->save()) {
                                                array_push($resultadoErrors, 'Error al guardar el Precio del Material con clave: '.$producto['2']);
                                            }
                                        }

                                    }
                                } else {
                                    array_push($resultadoErrors, 'Error al actualizar el Material con clave: '.$producto['2']);
                                }

                            } else {
                                //NO Existe el Material y crea material
                                $newProducto = new SolicitudCompraProducto();
                                $newProducto->solicitud_compra_id = $orden_compra->solicitud_compra_id;
                                $newProducto->producto_id = $producto['2'];
                                $newProducto->unidad_id = $producto['4'];
                                $newProducto->cantidad = $producto['6'];
                                $newProducto->area_de_uso = $producto['7'];
                                $newProducto->marca = $producto['8'];
                                $newProducto->modelo = $producto['9'];
                                $newProducto->observaciones = $producto['10'];
                                $newProducto->num_duplicado_order = $producto['16'];

                                if ($orden_compra->clave_version === Consts::ORDEN_COMPRA_V1) {
                                    $newProducto->precio_lista = $producto['11'];
                                    $newProducto->precio_unitario = $producto['12'];
                                    $newProducto->importe = $producto['13'];
                                    $newProducto->descuento = $producto['14'];
                                    $newProducto->subtotal = $producto['15'];
                                }

                                if ($newProducto->save()) {
                                    if ($orden_compra->clave_version === Consts::ORDEN_COMPRA_V2) {
                                        //Crear Precio del material
                                        $precio_material = new RelSolicitudCompraProductoPrecio();
                                        $precio_material->solicitud_compra_producto_id = $newProducto->id;
                                        $precio_material->orden_compra_id = $orden_compra->id;
                                        $precio_material->precio_lista = $producto['11'];
                                        $precio_material->precio_unitario = $producto['12'];
                                        $precio_material->importe = $producto['13'];
                                        $precio_material->descuento = $producto['14'];
                                        $precio_material->subtotal = $producto['15'];
                                        $precio_material->created_by = auth()->id();
                                        $precio_material->updated_by = auth()->id();
                                        if (!$precio_material->save()) {
                                            array_push($resultadoErrors, 'Error al actualizar el Precio del Material con clave: '.$producto['2']);
                                        }
                                    }
                                } else {
                                    array_push($resultadoErrors, 'Error al actualizar nuevo Material con clave: '.$producto['2']);
                                }
                            }

                            // Guardar el ID del producto procesado
                            $productos_ids[] = $producto['2'];
                        }

                        // 2. Buscar Materiales que no están en la lista actualizada para Eliminar su precio
                        $Materiales_Eliminar = SolicitudCompraProducto::where('solicitud_compra_id', $orden_compra->solicitud_compra_id)
                            ->whereNotIn('producto_id', $productos_ids) // Solo los que no estén en la lista
                            ->get();
                        if (!empty($Materiales_Eliminar)){
                            foreach ($Materiales_Eliminar as $material_borrar){
                                //Eliminar Precios de materiales
                                $material_precio_delete = RelSolicitudCompraProductoPrecio::where('solicitud_compra_producto_id', $material_borrar->id)->delete();
                            }
                        }
                        //Eliminar Materiales que no están en la lista actualizada
                        $Materiales_eliminar = SolicitudCompraProducto::where('solicitud_compra_id', $orden_compra->solicitud_compra_id)
                            ->whereNotIn('producto_id', $productos_ids)->delete(); // Solo los que no estén en la lista

                    } else if($orden_compra->clave_version === Consts::ORDEN_COMPRA_V2 && $orden_compra->estatus_id === $estatus_por_autorizar->id && !empty($orden_compra->solicitud_compra_clon_id)) {
                        //Actualizar O.C. en estatus Por autorizar y con CLON
                        $respuesta_update = $this->updateOrdenCompraMaterialesV2($orden_compra, $productos_data);
                        if ($respuesta_update['error']){
                            array_push($resultadoErrors, 'No se pudó actualizar la Orden de compra con sus Materiales clon.');
                        }
                    } else if($orden_compra->clave_version === Consts::ORDEN_COMPRA_V2 && $orden_compra->estatus_id === $estatus_autorizado->id && !empty($orden_compra->solicitud_compra_clon_id)) {
                        //Actualizar O.C. en estatus Autorizado y con CLON
                        $respuesta_update = $this->updateOrdenCompraMaterialesV2($orden_compra, $productos_data);
                        if ($respuesta_update['error']){
                            array_push($resultadoErrors, 'No se pudó actualizar la Orden de compra con sus Materiales clon.');
                        }
                    }

                    //====== Anticipos ======
                    //Dar de baja los Anticipos actuales
                    $anticipos_delete = OrdenCompraAnticipo::where('orden_compra_id', $orden_compra->id)->get();
                    if (!empty($anticipos_delete)) {
                        foreach ($anticipos_delete as $anticipo) {
                            $anticipo->delete();
                        }
                    }

                    //Lista-Anticipos
                    $anticipos_data = $dataAll['anticipos_data'];
                    if (!empty($anticipos_data)) {
                        foreach ($anticipos_data as $anticipo) {
                            $nuevo_anticipo = new OrdenCompraAnticipo();
                            $nuevo_anticipo->orden_compra_id = $orden_compra->id;
                            $nuevo_anticipo->comentario = $anticipo['1'];
                            $nuevo_anticipo->anticipo = $anticipo['2'];
                            $nuevo_anticipo->anticipo_letras = $anticipo['3'];
                            $nuevo_anticipo->forma_de_pago = $anticipo['4'];
                            if (!empty($anticipo['5'])){
                                $nuevo_anticipo->fecha_anticipo = (Carbon::createFromFormat('d/m/Y', $anticipo['5']))->format('Y-m-d');
                            } else {
                                $nuevo_anticipo->fecha_anticipo = Carbon::now()->format('Y-m-d');
                            }
                            $nuevo_anticipo->save();
                        }
                    }
                } else {
                    array_push($resultadoErrors, 'No se pudó actualizar la Orden de compra.');
                }
            }

            if (empty($resultadoErrors)) {
                DB::commit();
                $response = [
                    'status' => "OK",
                    'error' => false,
                    'message' => 'Orden de compra actualizada correctamente.',
                    'errors' => ''
                ];
            } else {
                DB::rollBack();
                $response = [
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'Ocurrió un error al actualizar la Orden de compra.',
                    'errors' => $resultadoErrors
                ];
            }

            return response()->json($response);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::info('Exception => ' . $e->getMessage() . ' in ' . $e->getLine());
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'message' => 'Ocurrio un error al actualizar la Orden de compra, contacte al administrador del sistema.',
                'errors' => 'Exception => ' . $e->getMessage() . ' in ' . $e->getLine()
            ]);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            //Estatus Por autorizar
            $estatus_por_autorizar = Estatus::where('clave', 'por_autorizar')->whereNull('deleted_at')->first();
            //Estatus: Autorizado
            $estatus_autorizado = Estatus::where('clave', 'autorizado')->whereNull('deleted_at')->first();

            $orden_compra = OrdenCompra::findOrFail($id);
            if ($orden_compra->estatus_id === $estatus_autorizado->id) {
                return response()->json(['error' => true, 'message' => 'No se puede eliminar la Orden de compra porque ya se encuentra en estatus Autorizado.']);
            }
            DB::beginTransaction();

            if ($orden_compra->clave_version === Consts::ORDEN_COMPRA_V2 && $orden_compra->estatus_id === $estatus_por_autorizar->id && !empty($orden_compra->solicitud_compra_clon_id)) {
                //Eliminar Clon
                $solicitud_compra_clon = SolicitudCompraClon::find($orden_compra->solicitud_compra_clon_id);
                SolicitudCompraProductoClon::where('solicitud_compra_clon_id', $solicitud_compra_clon->id)->delete();
                $solicitud_compra_clon->delete();
            }

            //En todas las versiones se Eliminan los Precios de materiales si los tiene
            RelSolicitudCompraProductoPrecio::where('orden_compra_id', $orden_compra->id)->delete();
            //En todas las versiones se Eliminan los anticipos si los tiene
            OrdenCompraAnticipo::where('orden_compra_id', $orden_compra->id)->delete();
            $orden_compra->delete();

            DB::commit();
            return response()->json(['error' => false, 'message' => 'Registro eliminado.']);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['error' => true, 'message' => 'No se pudó eliminar la Orden de compra.']);
        }
    }

    public function get_Lista_Ordenes_Compras(Request $request)
    {
        ini_set('memory_limit', '-1');
        if ($request->ajax()) {
            $ordenes_compras = OrdenCompra::with([
                'proveedor',
                'ivaImpuesto',
                'estatus',
                'ordenCompraAnticipo',
                'solicitudCompra',
                'solicitudCompra.obra',
                'solicitudCompra.ingeniero',
                'solicitudCompra.solicitudCompraProducto',
                'solicitudCompra.solicitudCompraProducto.producto',
                'solicitudCompra.solicitudCompraProducto.unidad',
                'tipoPago',
            ])
                ->whereNull('deleted_at')
                ->orderBy('id', 'desc')->get();

            $ordenes_compras->map(function ($ordenCompra) {
                $total = !empty($ordenCompra->total) ? $ordenCompra->total : 0;
                $ordenCompra->total = number_format($ordenCompra->total, 2);

                //Lista materiales
                $conceptos_materiales = "";
                $lista_productos = $ordenCompra->solicitudCompra->solicitudCompraProducto;
                foreach ($lista_productos as $key => $producto) {
                    $conceptos_materiales .= $producto->producto->nombre . ', ';
                }
                $conceptos_materiales = substr($conceptos_materiales, 0, -2);
                $conceptos_materiales .= ".";
                $ordenCompra->conceptos_materiales = $conceptos_materiales;

                //Lista de Anticipos
                $suma_array_anticipos = array();
                $lista_anticipos = $ordenCompra->ordenCompraAnticipo;
                if (!empty($lista_anticipos)) {
                    foreach ($lista_anticipos as $key => $anticipo) {
                        $suma_array_anticipos[$key] = floatval($anticipo->anticipo);
                        $anticipo->tipo_pago = $ordenCompra->tipoPago;

                        if ($key !== count($lista_anticipos)-1) {
                            $anticipo->es_ultimo = false;
                        } else {
                            $anticipo->es_ultimo = true;
                        }
                    }
                }
                $total_sum_anticipos = array_sum($suma_array_anticipos);
                $total_sum_anticipos_aux = !empty($total_sum_anticipos) ? $total_sum_anticipos : 0;
                $ordenCompra->total_sum_anticipos = number_format($total_sum_anticipos, 2);
                $ordenCompra->pendiente_por_pagar = number_format($total - $total_sum_anticipos_aux, 2);
            });

            return DataTables::of($ordenes_compras)->make(true);
        }
    }

    public function get_Lista_Orden_Select(Request $request)
    {
        $ordenes_compras = [];
        $filtrar_estatus = null;
        $enviado_a_pago = [];
        $orderBy = 'asc';

        //Estatus: Autorizado
        $estatus_autorizado = Estatus::where('clave', 'autorizado')->whereNull('deleted_at')->first();

        if ($request->has('type')) {
            $type = $request->type;
            switch ($type) {
                case 'enviar_a_pago_proveedor':
                    $filtrar_estatus = $estatus_autorizado->id; //Autorizado
                    $enviado_a_pago = [0];
                    break;
                case 'filtro_oc_enviado_a_pago':
                    $filtrar_estatus = $estatus_autorizado->id; //Autorizado
                    $enviado_a_pago = [1]; //Ordende compra enviado a: Pago a Proveedores
                    $orderBy = 'desc';
                    break;
                case 'filtro_oc_all':
                    $filtrar_estatus = null; //all Estatus para O.C.
                    $enviado_a_pago = [0,1]; //Ordende compra enviado a pago proveedor
                    $orderBy = 'desc';
                    break;
            }
        }

        if ($request->has('search')) {
            $search = $request->search;

            $ordenes_compras = OrdenCompra::select('id', 'codigo')
                ->when(!empty($filtrar_estatus), function ($query) use ($filtrar_estatus) {
                    $query->where('estatus_id', $filtrar_estatus);
                })
                ->whereIn('enviado_a_pago', $enviado_a_pago)
                ->where('codigo', 'LIKE', '%' . $search . '%')
                ->whereNull('deleted_at')
                ->orderBy('codigo', $orderBy)
                ->get();
        } else {
            $ordenes_compras = OrdenCompra::select('id', 'codigo')
                ->when(!empty($filtrar_estatus), function ($query) use ($filtrar_estatus) {
                    $query->where('estatus_id', $filtrar_estatus);
                })
                ->whereIn('enviado_a_pago', $enviado_a_pago)
                ->whereNull('deleted_at')
                ->orderBy('codigo', $orderBy)
                ->get();
        }

        if (!$ordenes_compras->isEmpty()) {
            $valid_tags = [];
            foreach ($ordenes_compras as $item) {
                $valid_tags[] = ['id' => $item->id, 'text' => $item->codigo];
            }
            return response()->json($valid_tags);
        } else {
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'message' => 'No se encontraron resultados'
            ]);
        }
    }

    public function delete_Archivo_Pdf($id)
    {
        $orden_compra = OrdenCompra::findOrFail($id);
        $name_pdf = $orden_compra->url_descarga_pdf;
        $orden_compra->url_descarga_pdf = '';
        $orden_compra->update();

        @unlink($this->path_archivo_pdf . '/' . $name_pdf);

        return redirect('/ordenes-de-compras/' . $id . '/edit')->withSuccess('Archivo PDF eliminado.');
    }

    private function getFoliador($ultimo_id = 0)
    {
        $foliador = 1;
        if ($ultimo_id > 0) {
            $foliador = str_pad($ultimo_id,  4, "0", STR_PAD_LEFT);
        } else {
            $foliador = str_pad($foliador,  4, "0", STR_PAD_LEFT);
        }
        return $foliador;
    }

    /**
     * Cambia el estatus de la O.C. a Autorizada
     * y clona la solicitud de compras junto con sus materiales
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function actualizar_Estatus_Autorizado(Request $request)
    {
        try {
            DB::beginTransaction();
            $response = [];
            $dataAll = $request->all();

            $validator = \Validator::make($dataAll, [
                'id_orden' => 'required',
                'estatus_id' => 'required',
            ]);

            if ($validator->fails()) {
                $response = [
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'Campos obligatorios',
                    'errors' => $validator->errors()->all()
                ];
            } else {
                $orden_compra = OrdenCompra::with([
                    'proveedor',
                    'ivaImpuesto',
                    'estatus',
                    'ordenCompraAnticipo',
                    'solicitudCompra',
                    'solicitudCompra.obra',
                    'solicitudCompra.ingeniero',
                    'solicitudCompra.solicitudCompraProducto',
                    'solicitudCompra.solicitudCompraProducto.producto',
                    'solicitudCompra.solicitudCompraProducto.unidad',
                ])
                    ->where('id', $dataAll['id_orden'])
                    ->whereNull('deleted_at')
                    ->first();

                if ($orden_compra->clave_version === Consts::ORDEN_COMPRA_V1) {
                    foreach ($orden_compra->solicitudCompra->solicitudCompraProducto as $material) {
                            if ($material->precio_lista == 0 || $material->precio_unitario == 0 || $material->importe == 0 || $material->subtotal == 0) {
                                DB::rollBack();
                                return response()->json([
                                    'status' => "ERROR",
                                    'error' => true,
                                    'message' => 'No se puede Autorizar la Orden de compra, debe agregar el precio del Material: '.$material->producto->nombre,
                                ]);
                            }
                    }
                } else if ($orden_compra->clave_version === Consts::ORDEN_COMPRA_V2) {
                    foreach ($orden_compra->solicitudCompra->solicitudCompraProducto as $material) {
                        $precio_del_material = RelSolicitudCompraProductoPrecio::where('solicitud_compra_producto_id', $material->id)
                            ->where('orden_compra_id', $orden_compra->id)
                            ->whereNull('deleted_at')->first();

                        if (empty($precio_del_material)) {
                            DB::rollBack();
                            return response()->json([
                                'status' => "ERROR",
                                'error' => true,
                                'message' => 'No se puede Autorizar la Orden de compra, debe agregar el precio del Material: '.$material->producto->nombre,
                            ]);
                        } else if ($precio_del_material->precio_lista == 0 || $precio_del_material->precio_unitario == 0 || $precio_del_material->importe == 0 || $precio_del_material->subtotal == 0) {
                            DB::rollBack();
                            return response()->json([
                                'status' => "ERROR",
                                'error' => true,
                                'message' => 'No se puede Autorizar la Orden de compra, debe agregar el precio del Material: '.$material->producto->nombre,
                            ]);
                        }
                    }
                }

                //Lista de Anticipos
                $suma_array_anticipos = array();
                $lista_anticipos = $orden_compra->ordenCompraAnticipo;
                if (!empty($lista_anticipos)) {
                    foreach ($lista_anticipos as $key => $anticipo) {
                        $suma_array_anticipos[$key] = floatval($anticipo->anticipo);
                    }
                }
                $total_sum_anticipos = array_sum($suma_array_anticipos);

                //Estatus: Autorizado
                $estatus_autorizado = Estatus::where('clave', 'autorizado')->whereNull('deleted_at')->first();

                //Si es Pago total valida sea igual importe total
                if ($orden_compra->tipo_pago_id === Consts::PAGO_TOTAL_ID) {
                    if (bccomp($total_sum_anticipos, $orden_compra->total, 2) == 0) {
                        //Permite cambiar de Estatus Autorizado
                        if ($orden_compra->update(['estatus_id' => $estatus_autorizado->id])) {

                            if ($orden_compra->clave_version === Consts::ORDEN_COMPRA_V1) {
                                $response = [
                                    'status' => "OK",
                                    'error' => false,
                                    'message' => 'Orden de compra Autorizada con éxito.',
                                ];
                            } else if ($orden_compra->clave_version === Consts::ORDEN_COMPRA_V2 && $orden_compra->solicitud_compra_clon_id !== null) {
                                //Ya existe el clon
                                $response = [
                                    'status' => "OK",
                                    'error' => false,
                                    'message' => 'Orden de compra Autorizada con éxito.',
                                ];
                            } else if ($orden_compra->clave_version === Consts::ORDEN_COMPRA_V2 && $orden_compra->solicitud_compra_clon_id == null) {
                                //Clonar S.C., sus Materiales con Precios y Comentarios Grupales
                                $respueta_clon_sc = $this->guardarClonSolicitudYMateriales($orden_compra, $orden_compra->solicitud_compra_id);
                                if ($respueta_clon_sc['error']) {
                                    $response = [
                                        'status' => "ERROR",
                                        'error' => true,
                                        'message' => $respueta_clon_sc['error_mensaje'],
                                    ];
                                } else {
                                    $response = [
                                        'status' => "OK",
                                        'error' => false,
                                        'message' => 'Orden de compra Autorizada con éxito.',
                                    ];
                                }
                            }

                        } else {
                            $response = [
                                'status' => "ERROR",
                                'error' => true,
                                'message' => 'No se puede Autorizar la Orden de compra.',
                            ];
                        }

                    } else {
                        $response = [
                            'status' => "ERROR",
                            'error' => true,
                            'message' => 'No se puede Autorizar, debe cubrir el monto total.',
                        ];
                    }
                } else if ($orden_compra->tipo_pago_id === Consts::PAGO_ANTICIPOS_ID) {
                    //Permite cambiar de Estatus Autorizado
                    if ($orden_compra->update(['estatus_id' => $estatus_autorizado->id])) {
                        if ($orden_compra->clave_version === Consts::ORDEN_COMPRA_V1) {
                            $response = [
                                'status' => "OK",
                                'error' => false,
                                'message' => 'Orden de compra Autorizada con éxito.',
                            ];
                        } else if ($orden_compra->clave_version === Consts::ORDEN_COMPRA_V2 && $orden_compra->solicitud_compra_clon_id !== null) {
                            //Ya existe el clon
                            $response = [
                                'status' => "OK",
                                'error' => false,
                                'message' => 'Orden de compra Autorizada con éxito.',
                            ];
                        } else if ($orden_compra->clave_version === Consts::ORDEN_COMPRA_V2 && $orden_compra->solicitud_compra_clon_id == null) {
                            //Clonar S.C., sus Materiales con Precios y Comentarios Grupales
                            $respueta_clon_sc = $this->guardarClonSolicitudYMateriales($orden_compra, $orden_compra->solicitud_compra_id);
                            if ($respueta_clon_sc['error']) {
                                $response = [
                                    'status' => "ERROR",
                                    'error' => true,
                                    'message' => $respueta_clon_sc['error_mensaje'],
                                ];
                            } else {
                                $response = [
                                    'status' => "OK",
                                    'error' => false,
                                    'message' => 'Orden de compra Autorizada con éxito.',
                                ];
                            }
                        }

                    } else {
                        $response = [
                            'status' => "ERROR",
                            'error' => true,
                            'message' => 'No se puede Autorizar la Orden de compra.',
                        ];
                    }

                }

            }

            if ($response['error']) {
                DB::rollBack();
            } else {
                DB::commit();
            }

            return response()->json($response);

        } catch (\Exception $ex) {
            DB::rollBack();
            Log::info('Exception: actualizar_Estatus_Autorizado => ' . $ex->getMessage() . ' in ' . $ex->getLine());
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'message' => 'Ocurrio un error inesperado, no se puede Autorizar la Orden de compra.',
            ]);
        }
    }

    /**
     * @param $id => clave de O.C.
     * @param $enviar_correo => Si es 1 envia PDF por correo
     * @param $solo_requisicion_compras => si es 0 imprime PDF O.C y R.C., Si es 1 imprime solo PDF R.C.
     * @return \Illuminate\Http\JsonResponse|void
     */
    public function print_pdf($id, $enviar_correo = 0, $solo_requisicion_compras = 0)
    {
        try {
            //Estatus Por autorizar
            $estatus_por_autorizar = Estatus::where('clave', 'por_autorizar')->whereNull('deleted_at')->first();
            //Estatus: Autorizado
            $estatus_autorizado = Estatus::where('clave', 'autorizado')->whereNull('deleted_at')->first();

            $orden_compra = OrdenCompra::with([
                'proveedor',
                'ivaImpuesto',
                'estatus',
                'ordenCompraAnticipo',
                'solicitudCompra',
                'solicitudCompra.obra',
                'solicitudCompra.ingeniero',
                'solicitudCompra.solicitudCompraProducto',
                'solicitudCompra.solicitudCompraProducto.producto',
                'solicitudCompra.solicitudCompraProducto.unidad',
                'solicitudCompra.solicitudCompraProducto.RelSolicitudProductoComentarioMaterial.comentariosMaterial',
                'SolicitudCompraClon',
                'SolicitudCompraClon.obra',
                'SolicitudCompraClon.ingeniero',
                'SolicitudCompraClon.SolicitudCompraProductoClon',
                'SolicitudCompraClon.SolicitudCompraProductoClon.producto',
                'SolicitudCompraClon.SolicitudCompraProductoClon.unidad',
                'SolicitudCompraClon.SolicitudCompraProductoClon.RelSolicitudProductoComentarioMaterialClon.ComentarioMaterialClon'
            ])->where('id', $id)
                ->whereNull('deleted_at')
                ->first();

            if ($orden_compra->clave_version === Consts::ORDEN_COMPRA_V2 && $orden_compra->estatus_id === $estatus_por_autorizar->id && $orden_compra->solicitud_compra_clon_id == null) {
                foreach ($orden_compra->solicitudCompra->solicitudCompraProducto as $material) {
                    //Obtiene el precio del material
                    $precio_del_material = RelSolicitudCompraProductoPrecio::where('solicitud_compra_producto_id', $material->id)
                        ->where('orden_compra_id', $orden_compra->id)
                        ->whereNull('deleted_at')->first();

                    //Setea el precio al material
                    $material->rel_solicitud_compra_producto_precio = $precio_del_material;
                }
            }

//            $pdf = PDFSnappy::loadView('admin.ordenes_de_compras.print_pdf', compact('orden_compra'));
//            $pdf->setOption('header-spacing', 1);
//            $pdf->setOption('header-html', view('admin.prints.prints_header'));
//            $pdf->setOption('footer-html', view('admin.prints.prints_footer'));
//            $pdf->setOptions([
//                'margin-top'    => 15,
//                'margin-bottom' => 15,
//                'margin-right'  => 15,
//                'margin-left'   => 15,
//            ]);

            $obra_mantenimiento = Consts::OBRA_MANTENIMIENTO;

            $pdf = App::make('dompdf.wrapper');
            $pdf->setOption(['dpi' => 110, 'defaultFont' => 'sans-serif', 'isPhpEnabled' => true]);
            $pdf->loadView('admin.ordenes_de_compras.print_pdf_oc', compact('orden_compra', 'estatus_por_autorizar', 'estatus_autorizado', 'obra_mantenimiento', 'solo_requisicion_compras'));

            $name_temp = ($solo_requisicion_compras === 0) ? '_temp_Orden_de_compra.pdf' : '_temp_Solicitud_de_compra.pdf';

            $nombre_pdf = $orden_compra->id . $name_temp;
            $pdf_oc_get = ($solo_requisicion_compras === 0) ? storage_path('app/public/pdf_oc_temp/').$nombre_pdf : storage_path('app/public/pdf_solicitud_temp/').$nombre_pdf;
            @unlink($pdf_oc_get);
            $pdf->save($pdf_oc_get);

            $pdfMerger = PDFMerger::init(); //Initialize the merger
            $pdfMerger->addPDF($pdf_oc_get, 'all');
            if ($orden_compra->url_descarga_pdf){
                $url_descarga_pdf = storage_path('app/public/').$orden_compra->url_descarga_pdf;
                $pdfMerger->addPDF($url_descarga_pdf, 'all');
            }
            $pdfMerger->merge();
            $pdfMerger->save($pdf_oc_get);

            if ($enviar_correo == 1) {
                if (!empty($orden_compra->emails)) {
                    $email_asunto = ($orden_compra->email_asunto) ? $orden_compra->email_asunto : 'Orden de Compra';
                    $msg = [
                        'subject' => $email_asunto,
                        'content' => $orden_compra->email_contenido,
                    ];
                    $emails_array = explode(',', $orden_compra->emails);
                    foreach ($emails_array as $email) {
                        Mail::to($email)->queue(new MessageReceived($orden_compra, $pdf_oc_get, $msg, 'emails.correo-orden-compra', 'Orden_de_compra.pdf'));
                    }
                } else {
                    $response = [
                        'status' => "ERROR",
                        'error' => true,
                        'message' => 'No se encontraron emails',
                    ];
                    return response()->json($response);
                }
            } else {
                return $pdfMerger->save($nombre_pdf, "download");
                //return $pdf->download('Orden_de_compra.pdf');
            }
            //return $pdf->stream();
        } catch (\Exception $e) {
            Log::info('Exception => ' . $e->getMessage() . ' in ' . $e->getLine());
            $response = [
                'status' => "ERROR",
                'error' => true,
                'message' => 'Ocurrio un error al generar el pdf, contacte al administrador del sistema.',
            ];
            return response()->json($response);
        }
    }

    /**
     * Subir un archivo PDF para una orden de compra.
     */
    public function uploadPDF(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:pdf|max:4096',
            'orden_id' => 'required|exists:ordenes_compras,id',
        ]);

        $file = $request->file('file');
        $uniqueName = uniqid(). '.' . $file->getClientOriginalExtension();
        $filePath = $file->storeAs('orden_compra_pdf', $uniqueName, 'public');
        $filePath = $this->path_archivo_pdf .'/'. $uniqueName;
        $file->move($this->path_archivo_pdf, $uniqueName);

        $orden = OrdenCompra::findOrFail($request->orden_id);
        $orden->url_descarga_pdf = $filePath;
        $orden->save();

        return response()->json([
            'message' => 'Archivo subido con éxito.',
            'path' => $filePath,
        ], 200);
    }

    /**
     * Actualizar información de una orden de compra, incluyendo el archivo PDF.
     */
    public function updateOrderWithPDF(Request $request, $id)
    {
        $orden = OrdenCompra::findOrFail($id);

        $request->validate([
            'file' => 'nullable|mimes:pdf|max:2048', // Solo PDF opcional
            'codigo' => 'nullable|string|max:255',
            'comentarios' => 'nullable|string',
        ]);

        if ($request->hasFile('file')) {
            // Eliminar el archivo anterior si existe
            if ($orden->url_descarga_pdf) {
                Storage::delete($orden->url_descarga_pdf);
            }

            // Subir el nuevo archivo
            $file = $request->file('file');
            $uniqueName = Str::uuid() . '.' . $file->getClientOriginalExtension();
            $path = $file->storeAs('ordenes_compras', $uniqueName);

            $orden->url_descarga_pdf = $path;
        }

        // Actualizar otros campos
        //$orden->codigo = $request->input('codigo', $orden->codigo);
        $orden->comentarios = $request->input('comentarios', $orden->comentarios);
        $orden->save();

        return response()->json([
            'success' => true,
            'message' => 'Orden de compra actualizada correctamente.',
        ]);
    }

    /**
     * Eliminar el archivo PDF asociado a una orden de compra.
     */
    public function deletePDF($id)
    {
        $orden = OrdenCompra::findOrFail($id);

        if ($orden->url_descarga_pdf) {
            // Eliminar el archivo del almacenamiento
            Storage::delete($orden->url_descarga_pdf);

            // Eliminar la referencia en la base de datos
            $orden->url_descarga_pdf = null;
            $orden->save();

            return response()->json([
                'success' => true,
                'message' => 'PDF eliminado correctamente.',
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'No se encontró un archivo PDF para esta orden.',
        ], 404);
    }

    public function print_pdf_list_selected(Request $request)
    {
        try {
            $dataAll = $request->all();
            $resultadoErrors = []; //Lista de Errores

            $validator = \Validator::make($dataAll[0], [
                'id' => 'required|integer',
            ]);

            if ($validator->fails()) {
                $response = [
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'Parámetros incompletos.',
                ];
            } else {
                $workflowIds = [];
                foreach ($dataAll as $item) {
                    array_push($workflowIds, $item['id']);
                }

                //Estatus Por autorizar
                $estatus_por_autorizar = Estatus::where('clave', 'por_autorizar')->whereNull('deleted_at')->first();
                //Estatus: Autorizado
                $estatus_autorizado = Estatus::where('clave', 'autorizado')->whereNull('deleted_at')->first();

                $ordenes_de_compras = OrdenCompra::with([
                    'proveedor',
                    'ivaImpuesto',
                    'estatus',
                    'ordenCompraAnticipo',
                    'solicitudCompra',
                    'solicitudCompra.obra',
                    'solicitudCompra.ingeniero',
                    'solicitudCompra.solicitudCompraProducto',
                    'solicitudCompra.solicitudCompraProducto.producto',
                    'solicitudCompra.solicitudCompraProducto.unidad',
                    'solicitudCompra.solicitudCompraProducto.RelSolicitudProductoComentarioMaterial.comentariosMaterial',
                    'SolicitudCompraClon',
                    'SolicitudCompraClon.obra',
                    'SolicitudCompraClon.ingeniero',
                    'SolicitudCompraClon.SolicitudCompraProductoClon',
                    'SolicitudCompraClon.SolicitudCompraProductoClon.producto',
                    'SolicitudCompraClon.SolicitudCompraProductoClon.unidad',
                    'SolicitudCompraClon.SolicitudCompraProductoClon.RelSolicitudProductoComentarioMaterialClon.ComentarioMaterialClon'
                ])
                    ->whereIn('id', $workflowIds)
                    ->whereNull('deleted_at')
                    ->orderBy('id', 'desc')
                    ->get();

                foreach ($ordenes_de_compras as $orden_compra) {
                    if ($orden_compra->clave_version === Consts::ORDEN_COMPRA_V2 && $orden_compra->estatus_id === $estatus_por_autorizar->id && $orden_compra->solicitud_compra_clon_id == null) {
                        foreach ($orden_compra->solicitudCompra->solicitudCompraProducto as $material) {
                            //Obtiene el precio del material
                            $precio_del_material = RelSolicitudCompraProductoPrecio::where('solicitud_compra_producto_id', $material->id)
                                ->where('orden_compra_id', $orden_compra->id)
                                ->whereNull('deleted_at')->first();

                            //Setea el precio al material
                            $material->rel_solicitud_compra_producto_precio = $precio_del_material;
                        }
                    }
                }

                if (env('APP_ENV') == 'production'){
                    #$path_public = '/home/grupoari/appordenes.grupoarista.com.mx/orden_compra_pdf_listado_temporal/';
                    $path_public = public_path('orden_compra_pdf_listado_temporal/');
                } else {
                    $path_public = public_path('orden_compra_pdf_listado_temporal/');
                }

                $obra_mantenimiento = Consts::OBRA_MANTENIMIENTO;
                //generar lista pdf
                $pdf = App::make('dompdf.wrapper');
                $pdf->setOption(['dpi' => 110, 'defaultFont' => 'sans-serif', 'isPhpEnabled' => true]);
                $pdf->loadView('admin.ordenes_de_compras.print_pdf_lista_ordenes_compras', compact('ordenes_de_compras', 'estatus_por_autorizar', 'estatus_autorizado', 'obra_mantenimiento'));

                $file_name = time() . '_Ordenes_de_compras.pdf';
                $pdf_oc_get = storage_path('app/public/orden_compra_pdf_listado_temporal/').$file_name;
                $pdf->save($pdf_oc_get);

                $pdfMerger = PDFMerger::init(); //Initialize the merger
                $pdfMerger->addPDF($pdf_oc_get, 'all');
                foreach ($ordenes_de_compras as $orden_compra) {
                    if ($orden_compra->url_descarga_pdf){
                        $url_descarga_pdf = storage_path('app/public/').$orden_compra->url_descarga_pdf;
                        $pdfMerger->addPDF($url_descarga_pdf, 'all');
                    }
                }
                $pdfMerger->merge();
                $pdfMerger->save($path_public.$file_name);

                $path = 'orden_compra_pdf_listado_temporal/';
                $archivo_url = $path . $file_name;

                @unlink($pdf_oc_get);

                $response = [
                    'status' => "OK",
                    'error' => false,
                    'message' => 'PDF generado con éxito.',
                    'archivo_url' => $archivo_url,
                ];
            }
            return response()->json($response);

        } catch (\Exception $e) {
            Log::info('Exception => ' . $e->getMessage() . ' in ' . $e->getLine());
            $response = [
                'status' => "ERROR",
                'error' => true,
                'message' => 'Ocurrio un error al generar el pdf, contacte al administrador del sistema.',
            ];
            return response()->json($response);
        }
    }

    /**
     * Cambiar el Estatus de la O.C. a Por autorizar
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function actualizar_estatus_por_autorizar(Request $request)
    {
        $response = [];
        $dataAll = $request->all();

        $validator = \Validator::make($dataAll, [
            'id_orden' => 'required',
            'estatus_id' => 'required',
        ]);

        if ($validator->fails()) {
            $response = [
                'status' => "ERROR",
                'error' => true,
                'message' => 'Campos obligatorios',
                'errors' => $validator->errors()->all()
            ];
        } else {
            $orden_compra = OrdenCompra::where('id', $dataAll['id_orden'])
                ->whereNull('deleted_at')
                ->first();

            //Estatus Por autorizar
            $estatus_por_autorizar = Estatus::where('clave', 'por_autorizar')->whereNull('deleted_at')->first();

            if ($orden_compra->update(['estatus_id' => $estatus_por_autorizar->id])) {
                $response = [
                    'status' => "OK",
                    'error' => false,
                    'message' => 'Orden de compra Autorizada con éxito.',
                ];
            } else {
                $response = [
                    'status' => "ERROR",
                    'error' => true,
                    'message' => 'No se puede Autorizar la Orden de compra.',
                ];
            }
        }

        return response()->json($response);
    }

    /**
     * Guardar Clon de Solicitud de compras para la O.C. Autorizada
     * @param $solicitud_compra_id
     * @return void
     */
    private function guardarClonSolicitudYMateriales($orden_compra, $solicitud_compra_id)
    {
        try {
            $response = [];

            //Solicitud de Compra Original
            $solicitud_compra = SolicitudCompra::with([
                'solicitudCompraProducto',
                'solicitudCompraProducto.RelSolicitudProductoComentarioMaterial.comentariosMaterial',
            ])->where('id', $solicitud_compra_id)->first();

            //Estatus: finalizado
            $estatus_finalizado = Estatus::where('clave', 'finalizado')->whereNull('deleted_at')->first();

            //Clon Solicitud
            $solicitud_compra_clon = new SolicitudCompraClon();
            $solicitud_compra_clon->obra_id = $solicitud_compra->obra_id;
            $solicitud_compra_clon->ingeniero_id = $solicitud_compra->ingeniero_id;
            $solicitud_compra_clon->observaciones = $solicitud_compra->observaciones;
            $solicitud_compra_clon->fecha_solicitud = $solicitud_compra->fecha_solicitud;
            $solicitud_compra_clon->fecha_entrega = $solicitud_compra->fecha_entrega;
            $solicitud_compra_clon->estatus_id = $estatus_finalizado->id;
            $solicitud_compra_clon->emails = $solicitud_compra->emails;
            $solicitud_compra_clon->email_asunto = $solicitud_compra->email_asunto;
            $solicitud_compra_clon->email_contenido = $solicitud_compra->email_contenido;
            $solicitud_compra_clon->codigo = $solicitud_compra->codigo;
            $solicitud_compra_clon->codigo_nombre = $solicitud_compra->codigo_nombre;
            $solicitud_compra_clon->clave_version = $solicitud_compra->clave_version;

            if ($solicitud_compra_clon->save()) {
                //Actualiza O.C. con la clave de la S.C. clonada
                if (!$orden_compra->update(['solicitud_compra_clon_id' => $solicitud_compra_clon->id])) {
                    return ['error' => true, 'error_mensaje' => 'Ocurrio un error al Actualizar la O.C. con la clave de la S.C. clonada.'];
                } else {
                    //Lista de Materiales origen
                    foreach ($solicitud_compra->solicitudCompraProducto as $producto) {
                        //obtener el precio del Material
                        $precio_del_material = RelSolicitudCompraProductoPrecio::where('solicitud_compra_producto_id', $producto->id)
                            ->where('orden_compra_id', $orden_compra->id)
                            ->whereNull('deleted_at')->first();

                        //Clon Materiales
                        $solicitud_compra_producto_clon = new SolicitudCompraProductoClon();
                        $solicitud_compra_producto_clon->solicitud_compra_clon_id = $solicitud_compra_clon->id;
                        $solicitud_compra_producto_clon->solicitud_compra_producto_id_origen = $producto->id;
                        $solicitud_compra_producto_clon->producto_id = $producto->producto_id;
                        $solicitud_compra_producto_clon->unidad_id = $producto->unidad_id;
                        $solicitud_compra_producto_clon->num_duplicado_order = $producto->num_duplicado_order;
                        $solicitud_compra_producto_clon->cantidad = $producto->cantidad;
                        $solicitud_compra_producto_clon->area_de_uso = $producto->area_de_uso;
                        $solicitud_compra_producto_clon->marca = $producto->marca;
                        $solicitud_compra_producto_clon->modelo = $producto->modelo;
                        $solicitud_compra_producto_clon->observaciones = $producto->observaciones;
                        //Precio del material
                        $solicitud_compra_producto_clon->precio_lista = $precio_del_material->precio_lista;
                        $solicitud_compra_producto_clon->precio_unitario = $precio_del_material->precio_unitario;
                        $solicitud_compra_producto_clon->importe = $precio_del_material->importe;
                        $solicitud_compra_producto_clon->descuento = $precio_del_material->descuento;
                        $solicitud_compra_producto_clon->subtotal = $precio_del_material->subtotal;

                        if (!$solicitud_compra_producto_clon->save()) {
                            return ['error' => true, 'error_mensaje' => 'Ocurrio un error al clonar los materiales de la solicitud compra.'];
                        } else {

                            if (!empty($producto->RelSolicitudProductoComentarioMaterial)) {

                                $buscar_comentario_material_clon = ComentarioMaterialClon::where('solicitud_compra_id', $solicitud_compra_id)
                                    ->where('solicitud_compra_productos_ids', $producto->RelSolicitudProductoComentarioMaterial->comentariosMaterial->solicitud_compra_productos_ids)
                                    ->first();

                                if (empty($buscar_comentario_material_clon)) {
                                    //Si no existe - Crear Material clon
                                    $comentario_clon = new ComentarioMaterialClon();
                                    $comentario_clon->comentario = $producto->RelSolicitudProductoComentarioMaterial->comentariosMaterial->comentario;
                                    $comentario_clon->solicitud_compra_productos_ids = $producto->RelSolicitudProductoComentarioMaterial->comentariosMaterial->solicitud_compra_productos_ids;
                                    $comentario_clon->solicitud_compra_id = $solicitud_compra_id;
                                    $comentario_clon->solicitud_compra_productos_clon_ids = $solicitud_compra_producto_clon->id;
                                    $comentario_clon->solicitud_compra_clon_id = $solicitud_compra_clon->id;

                                    if ($comentario_clon->save()) {
                                        //Clona rel del comentario grupal
                                        $new_rel_materiales_comentario_clon = new RelSolicitudProductoComentarioMaterialClon();
                                        $new_rel_materiales_comentario_clon->solicitud_compra_producto_id_origen = $producto->RelSolicitudProductoComentarioMaterial->solicitud_compra_producto_id;
                                        $new_rel_materiales_comentario_clon->solicitud_compra_id = $solicitud_compra_id;
                                        $new_rel_materiales_comentario_clon->solicitud_compras_productos_clon_id = $solicitud_compra_producto_clon->id;
                                        $new_rel_materiales_comentario_clon->comentarios_materiales_clon_id = $comentario_clon->id;
                                        $new_rel_materiales_comentario_clon->users_id = Auth::id();
                                        $new_rel_materiales_comentario_clon->solicitud_compra_clon_id = $solicitud_compra_clon->id;
                                        if ($new_rel_materiales_comentario_clon->save()) {
                                            $response = ['error' => false, 'error_mensaje' => '', 'exito_mensaje' => 'Solicitud de Compra Clonada Correctamente.'];
                                        } else {
                                            return ['error' => true, 'error_mensaje' => 'Ocurrio un error al clonar el Rel comentario grupal del material para la Solicitud compra.'];
                                        }

                                    } else {
                                        return ['error' => true, 'error_mensaje' => 'Ocurrio un error al clonar el comentario grupal del material para la Solicitud compra.'];
                                    }
                                } else {
                                    //Si ya existe el comentario material clon, solo Crea rel material comentario clon
                                    $new_rel_materiales_comentario_clon = new RelSolicitudProductoComentarioMaterialClon();
                                    $new_rel_materiales_comentario_clon->solicitud_compra_producto_id_origen = $producto->RelSolicitudProductoComentarioMaterial->solicitud_compra_producto_id;
                                    $new_rel_materiales_comentario_clon->solicitud_compra_id = $solicitud_compra_id;
                                    $new_rel_materiales_comentario_clon->solicitud_compras_productos_clon_id = $solicitud_compra_producto_clon->id;
                                    $new_rel_materiales_comentario_clon->comentarios_materiales_clon_id = $buscar_comentario_material_clon->id;
                                    $new_rel_materiales_comentario_clon->users_id = Auth::id();
                                    $new_rel_materiales_comentario_clon->solicitud_compra_clon_id = $solicitud_compra_clon->id;
                                    if ($new_rel_materiales_comentario_clon->save()) {
                                        $response = ['error' => false, 'error_mensaje' => '', 'exito_mensaje' => 'Solicitud de Compra Clonada Correctamente.'];
                                    } else {
                                        return ['error' => true, 'error_mensaje' => 'Ocurrio un error al clonar el Rel comentario grupal del material para la Solicitud compra.'];
                                    }
                                }

                            } else {
                                $response = ['error' => false, 'error_mensaje' => '', 'exito_mensaje' => 'Solicitud de Compra Clonada Correctamente.'];
                            }

                        }
                    }
                }

            } else {
                $response = ['error' => true, 'error_mensaje' => 'Ocurrio un error al guardar la solicitud compra clon.'];
            }

            return $response;

        } catch (\Exception $ex) {
            Log::info('Exception: guardarClonSolicitudYMateriales => ' . $ex->getMessage() . ' in ' . $ex->getLine());
            return ['error' => true, 'error_mensaje' => 'Ocurrio un error inesperado al guardar la solicitud compra clon.'];
        }
    }

    /**
     * Eliminar Material y Precio de la BD
     * @return void
     */
    public function eliminarMaterialYPrecio($solicitud_compra_id, $producto_id, $orden_compra_id)
    {
        try {
            DB::beginTransaction();
            $response = [];

            if ($orden_compra_id) {
                $material_delete = SolicitudCompraProducto::where('solicitud_compra_id', $solicitud_compra_id)
                    ->where('producto_id', $producto_id)->first();
                if ($material_delete) {
                    $material_precio_delete = RelSolicitudCompraProductoPrecio::where('solicitud_compra_producto_id', $material_delete->id)
                        ->where('orden_compra_id', $orden_compra_id)->delete();

                    if ($material_delete->delete()) {
                        DB::commit();
                        $response = ['error' => false, 'error_mensaje' => '', 'exito_mensaje' => 'El Material y su precio ha sido eliminado.'];
                    } else {
                        DB::rollBack();
                        $response =  ['error' => true, 'error_mensaje' => 'Error al eliminar el Material.'];
                    }
                }
            } else {
                $material_delete = SolicitudCompraProducto::where('solicitud_compra_id', $solicitud_compra_id)
                    ->where('producto_id', $producto_id)->first();

                if ($material_delete) {
                    if ($material_delete->delete()) {
                        DB::commit();
                        $response = ['error' => false, 'error_mensaje' => '', 'exito_mensaje' => 'El Material ha sido eliminado con éxito.'];
                    } else {
                        DB::rollBack();
                        $response =  ['error' => true, 'error_mensaje' => 'Error al eliminar el Material.'];
                    }
                } else {
                    DB::rollBack();
                    $response = ['error' => false, 'error_mensaje' => '', 'exito_mensaje' => 'El Material ha sido borrado.'];
                }
            }

            return response()->json($response);

        } catch (\Exception $ex) {
            DB::rollBack();
            Log::info('Exception: eliminarMaterialYPrecio => ' . $ex->getMessage() . ' in ' . $ex->getLine());
            return response()->json([
                'error' => true,
                'message' => 'Error al eliminar el Material y su precio.',
                'error_mensaje' => $ex->getMessage(),
            ]);
        }
    }

    /**
     * Actualizar O.C. V2 en estatus Por autorizar y con CLON
     * @param $orden_compra
     * @param $data_materiales
     * @return array
     */
    private function updateOrdenCompraMaterialesV2($orden_compra, $data_materiales)
    {
        try {
            $response = [];
            if (count($data_materiales) > 0) {
                foreach ($data_materiales as $material) {
                    $existeMaterial = SolicitudCompraProductoClon::where('solicitud_compra_clon_id', $orden_compra->solicitud_compra_clon_id)
                        ->where('producto_id', $material['2'])
                        ->where('num_duplicado_order', $material['16'])
                        ->whereNull('deleted_at')
                        ->first();

                    if ($existeMaterial) {
                        //Actualiza Material existente
                        $existeMaterial->unidad_id = $material['4'];
                        $existeMaterial->cantidad = $material['6'];
                        $existeMaterial->area_de_uso = $material['7'];
                        $existeMaterial->marca = $material['8'];
                        $existeMaterial->modelo = $material['9'];
                        $existeMaterial->observaciones = $material['10'];
                        //Actualiza Precio para O.C. V2 y con CLON Material
                        $existeMaterial->precio_lista = $material['11'];
                        $existeMaterial->precio_unitario = $material['12'];
                        $existeMaterial->importe = $material['13'];
                        $existeMaterial->descuento = $material['14'];
                        $existeMaterial->subtotal = $material['15'];
                        $existeMaterial->num_duplicado_order = $material['16'];


                        if ($existeMaterial->update()) {
                            $response = [
                                'error' => false,
                                'message' => 'Se actualizó el Material con éxito.',
                            ];
                        } else {
                            return [
                                'error' => true,
                                'message' => 'Error al actualizar el Material con clave: '.$material['2'],
                            ];
                        }
                    } else {
                        //NO Existe el Material y crea material en CLON
                        $newMaterial = new SolicitudCompraProductoClon();
                        $newMaterial->solicitud_compra_clon_id = $orden_compra->solicitud_compra_clon_id;
                        $newMaterial->solicitud_compra_producto_id_origen = null;
                        $newMaterial->producto_id = $material['2'];
                        $newMaterial->unidad_id = $material['4'];
                        $newMaterial->cantidad = $material['6'];
                        $newMaterial->area_de_uso = $material['7'];
                        $newMaterial->marca = $material['8'];
                        $newMaterial->modelo = $material['9'];
                        $newMaterial->observaciones = $material['10'];
                        //Precio para O.C. V2 y con CLON Material
                        $newMaterial->precio_lista = $material['11'];
                        $newMaterial->precio_unitario = $material['12'];
                        $newMaterial->importe = $material['13'];
                        $newMaterial->descuento = $material['14'];
                        $newMaterial->subtotal = $material['15'];
                        $newMaterial->num_duplicado_order = $material['16'];

                        if ($newMaterial->save()) {
                            $response = [
                                'error' => false,
                                'message' => 'Se agregó el Material con éxito.',
                            ];
                        } else {
                            return [
                                'error' => true,
                                'message' => 'Error al agregar el Material con clave: '.$material['2'],
                            ];
                        }
                    }

                    // Guardar el ID del producto procesado
                    $productos_ids[] = $material['2'];
                }

                //Eliminar Materiales que no están en la lista actualizada
                $Materiales_eliminar = SolicitudCompraProductoClon::where('solicitud_compra_clon_id', $orden_compra->solicitud_compra_clon_id)
                    ->whereNotIn('producto_id', $productos_ids)->delete(); // Solo los que no estén en la lista

            } else {
                $response = [
                    'error' => true,
                    'message' => 'Error no hay materiales para actualizar la Orden de Compra.',
                ];
            }
            return $response;
        } catch (\Exception $e) {
            Log::info('Exception en updateOrdenCompraMaterialesV2 => ' . $e->getMessage() . ' in ' . $e->getLine());
            return [
                'error' => true,
                'message' => 'Ocurrió un error al actializar los materiales con clon para la Orden de Compra v2.',
            ];
        }
    }

}
