<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Obra;
use Illuminate\Http\Request;
use DataTables;
use Illuminate\Support\Facades\Auth;

class ObrasController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();

        $catalogo_de_obras_edit = $user->can('catalogo_de_obras_edit');
        $catalogo_de_obras_destroy = $user->can('catalogo_de_obras_destroy');

        return view('admin.catalogos.obras.lista', compact('catalogo_de_obras_edit', 'catalogo_de_obras_destroy'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('admin.catalogos.obras.nuevo');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'nombre' => 'required|min:3',
            'no_control' => 'required',
        ]);

        $obra = new Obra();
        $obra->nombre = $request->input('nombre');
        $obra->no_control = $request->input('no_control');
        $obra->direccion = $request->input('direccion');
        $obra->comentarios = $request->input('comentarios');
        $obra->save();

        return redirect()->route('catalogo-de-obras.index')->withSuccess('Obra guardada con éxito.');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $obra = Obra::findOrFail($id);
        return view('admin.catalogos.obras.editar', compact("obra"));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validated = $request->validate([
            'nombre' => 'required|min:3',
            'no_control' => 'required',
        ]);

        $obra = Obra::findOrFail($id);
        $obra->nombre = $request->input('nombre');
        $obra->no_control = $request->input('no_control');
        $obra->direccion = $request->input('direccion');
        $obra->comentarios = $request->input('comentarios');
        $obra->update();

        return redirect()->route('catalogo-de-obras.index')->withSuccess('Obra actualizada con éxito.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $obra = Obra::findOrFail($id);
            $obra->delete();

            return response()->json(['error' => false, 'message' => 'Registro eliminado.']);
        } catch (\Exception $ex) {
            return response()->json([
                'error' => true,
                'message' => $ex->getMessage(),
            ]);
        }
    }

    public function get_Lista_Obras(Request $request)
    {
        if ($request->ajax()) {
            $obras = Obra::whereNull('deleted_at')
                ->orderBy('id', 'desc')->get();

            return DataTables::of($obras)->make(true);
        }
    }

    public function get_Lista_Obras_Select(Request $request)
    {
        $obras = [];
        if($request->has('search')){
            $search = $request->search;

            $obras = Obra::select('id', 'nombre')
                ->whereNull('deleted_at')
                ->where('nombre', 'LIKE', '%'.$search.'%')
                ->orderBy('nombre', 'asc')
                ->get();
        }else{
            $obras = Obra::select('id', 'nombre')
                ->whereNull('deleted_at')
                ->orderBy('nombre', 'asc')
                ->get();
        }

        if(!$obras->isEmpty()){
            $valid_tags = [];
            foreach ($obras as $item) {
                $valid_tags[] = ['id' => $item->id, 'text' => $item->nombre];
            }
            return response()->json($valid_tags);
        }else{
            return response()->json([
                'status' => "ERROR",
                'error' => true,
                'message' => 'No se encontraron resultados'
            ]);
        }
    }

}
